<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\View;
use Illuminate\Support\Facades\DB;

use App\Models\State; // using your existing model file that contains extra service methods
class ExtraServiceController extends Controller
{
    // List services
    public function index()
    {
        if (!session()->has('admin_id')) {
            return redirect()->route('admin.login');
        }

        $services = state::fetchAllExtraServices();

        $pageHead = View::make('template.page_head')->render();
        $sidebar  = View::make('template.sidebar')->render();
        $pageFoot = View::make('template.page_foot')->render();

        $table = View::make('extra_services.listing', compact('services'))->render();

        return response($pageHead . $sidebar . $table . $pageFoot);
    }

    // Show add form
    public function create()
    {
        if (!session()->has('admin_id')) {
            return redirect()->route('admin.login');
        }

        $pageHead = View::make('template.page_head')->render();
        $sidebar  = View::make('template.sidebar')->render();
        $pageFoot = View::make('template.page_foot')->render();
        $form = View::make('extra_services.form')->render();

        return response($pageHead . $sidebar . $form . $pageFoot);
    }

    // Store new service
    public function store(Request $request)
    {
        $request->validate([
            'name' => 'required|string|max:255|unique:extra_services,name',
            'price' => 'required|numeric|min:0',
            'price_type' => 'required|in:per_day,one_time',
            'status' => 'required|in:active,inactive',
            'description' => 'nullable|string',
        ]);

        state::createExtraService($request->only(['name','price','price_type','status','description']));

        return redirect()->route('extra-services.index')->with('success','Service added successfully');
    }

    // Show edit form
    public function edit($id)
    {
        if (!session()->has('admin_id')) {
            return redirect()->route('admin.login');
        }

        $service = state::fetchExtraServiceById($id);

        $pageHead = View::make('template.page_head')->render();
        $sidebar  = View::make('template.sidebar')->render();
        $pageFoot = View::make('template.page_foot')->render();
        $form = View::make('extra_services.form', compact('service'))->render();

        return response($pageHead . $sidebar . $form . $pageFoot);
    }

    // Update service
    public function update(Request $request, $id)
    {
        $request->validate([
            'name' => 'required|string|max:255|unique:extra_services,name,' . $id,
            'price' => 'required|numeric|min:0',
            'price_type' => 'required|in:per_day,one_time',
            'status' => 'required|in:active,inactive',
            'description' => 'nullable|string',
        ]);

        state::updateExtraService($id, $request->only(['name','price','price_type','status','description']));

        return redirect()->route('extra-services.index')->with('success','Service updated successfully');
    }

    // Deactivate service (soft-like)
    public function destroy($id)
    {
        state::deactivateExtraService($id);

        return redirect()->route('extra-services.index')->with('success','Service deactivated successfully');
    }
// Admin: tickets listing (admin index)
public function ticketsIndex()
{
    if (!session()->has('admin_id')) {
        return redirect()->route('admin.login');
    }

    $tickets = DB::table('tickets')->orderBy('id', 'desc')->get();

    $pageHead = View::make('template.page_head')->render();
    $sidebar  = View::make('template.sidebar')->render();
    $pageFoot = View::make('template.page_foot')->render();

    $table = View::make('extra_services.tickets_listing', compact('tickets'))->render();

    return response($pageHead . $sidebar . $table . $pageFoot);
}

// Admin: view single ticket and action form
public function ticketShow($id)
{
    if (!session()->has('admin_id')) {
        return redirect()->route('admin.login');
    }

    $ticket = DB::table('tickets')->where('id', $id)->first();
    if (!$ticket) {
        return redirect()->route('admin.tickets.index')->with('error', 'Ticket not found');
    }

    $pageHead = View::make('template.page_head')->render();
    $sidebar  = View::make('template.sidebar')->render();
    $pageFoot = View::make('template.page_foot')->render();

    $view = View::make('extra_services.ticket_view', compact('ticket'))->render();

    return response($pageHead . $sidebar . $view . $pageFoot);
}

// Admin: update ticket (change status, optionally reply)
public function ticketUpdate(Request $request, $id)
{
    if (!session()->has('admin_id')) {
        return redirect()->route('admin.login');
    }

    $request->validate([
        'status' => 'required|in:active,closed',
        'admin_reply' => 'nullable|string',
    ]);

    $updateData = [
        'status' => $request->input('status'),
        'updated_at' => now(),
    ];

    DB::table('tickets')->where('id', $id)->update($updateData);

    // Optionally you can store admin replies in a separate table.
    // For now we append admin reply into a simple log column or ignore it.
    // If you want replies saved, create a ticket_replies table. (I can add that.)

    return redirect()->route('admin.tickets.show', $id)->with('success','Ticket updated successfully');
}

public function settingsIndex()
{
    if (!session()->has('admin_id')) {
        return redirect()->route('admin.login');
    }

    // Fetch required settings as key => value
    $settings = DB::table('settings')
        ->whereIn('meta_key', [
            'referral_income',
            'first_booking_income',
            'admin_commission'
        ])
        ->pluck('meta_value', 'meta_key');

    // Load common admin layout parts
    $pageHead = View::make('template.page_head')->render();
    $sidebar  = View::make('template.sidebar')->render();
    $pageFoot = View::make('template.page_foot')->render();

    // Load settings form view
    $content = View::make('extra_services.settings', compact('settings'))->render();

    return response($pageHead . $sidebar . $content . $pageFoot);
}

public function saveSettings(Request $request)
{
    if (!session()->has('admin_id')) {
        return redirect()->route('admin.login');
    }

    $request->validate([
        'referral_income'      => 'required|numeric|min:0|max:100',
        'first_booking_income' => 'required|numeric|min:0|max:100',
        'admin_commission'     => 'required|numeric|min:0|max:100',
    ]);

    foreach ($request->except('_token') as $key => $value) {
        DB::table('settings')->updateOrInsert(
            ['meta_key' => $key],
            ['meta_value' => $value]
        );
    }

    return back()->with('success', 'Settings updated successfully');
}

}
