<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\View;
use App\Models\State;
use App\Models\Merchant;
use Illuminate\Support\Facades\Storage;
use Illuminate\Validation\Rule;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Hash;


use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Str;
use Carbon\Carbon;

class MerchantController extends Controller
{
    // List cities


public function index(Request $request)
{
    if (!session()->has('merchant_id')) {
        return redirect()->route('merchant.login');
    }

    // merchant session
    $merchant = [
        'user_id'   => session('merchant_id'),
        'name'      => session('merchant_name'),
        'email'     => session('merchant_email'),
        'user_type' => session('user_type'),
    ];

    $merchantId = $merchant['user_id'];

    // WALLET BALANCE
    $walletBalance = DB::table('user_wallet')->where('user_id', $merchantId)->value('balance');
    $walletBalance = $walletBalance !== null ? (float)$walletBalance : 0.00;

    // WITHDRAWALS: counts by status
    $withdrawalsPending = DB::table('withdrawals')
        ->where('user_id', $merchantId)
        ->where('status', 'pending')
        ->count();

    $withdrawalsApproved = DB::table('withdrawals')
        ->where('user_id', $merchantId)
        ->where('status', 'approved')
        ->count();

    $withdrawalsRejected = DB::table('withdrawals')
        ->where('user_id', $merchantId)
        ->where('status', 'rejected')
        ->count();

    // ACTIVE PROPERTIES (owned by merchant)
    $activePropertiesCount = DB::table('properties')
        ->where(function($q) use ($merchantId) {
            $q->where('admin_id', $merchantId)->orWhere('user_id', $merchantId);
        })
        ->where('status', 'active')
        ->count();

    // ACTIVE BOOKINGS: bookings on merchant's active properties
    $myPropertyIds = DB::table('properties')
        ->where(function($q) use ($merchantId) {
            $q->where('admin_id', $merchantId)->orWhere('user_id', $merchantId);
        })
        ->where('status', 'active')
        ->pluck('id')
        ->toArray();

    $activeBookingsCount = 0;
    if (!empty($myPropertyIds)) {
        // define active bookings as booking_status = 'approved' (change if needed)
        $activeBookingsCount = DB::table('my_bookings')
            ->whereIn('property_id', $myPropertyIds)
            ->where('booking_status', 'approved')
            ->count();
    }

    // Render layout pieces (your existing header/sidebar/footer pattern)
    $pageHead = View::make('template.page_head_login', compact('merchant'))->render();
    $sidebar  = View::make('template.page_login_sidebar', compact('merchant'))->render();
    $pageFoot = View::make('template.page_foot_login')->render();

    // render dashboard view (below)
    $content = View::make('merchant_panel.merchant_dashboard', [
        'merchant' => $merchant,
        'walletBalance' => $walletBalance,
        'withdrawalsPending' => $withdrawalsPending,
        'withdrawalsApproved' => $withdrawalsApproved,
        'withdrawalsRejected' => $withdrawalsRejected,
        'activePropertiesCount' => $activePropertiesCount,
        'activeBookingsCount' => $activeBookingsCount,
    ])->render();

    return response($pageHead . $sidebar . $content . $pageFoot);
}

public function calendarBookings(Request $request)
{
    // merchant session check
    if (!session()->has('merchant_id')) {
        return response()->json([]);
    }

    $merchantId = session('merchant_id');

    // 1. Get merchant active properties
    $propertyIds = DB::table('properties')
        ->where('status', 'active')
        ->where(function ($q) use ($merchantId) {
            $q->where('admin_id', $merchantId)
              ->orWhere('user_id', $merchantId);
        })
        ->pluck('id');

    if ($propertyIds->isEmpty()) {
        return response()->json([]);
    }

    // 2. Get active + approved bookings
    $bookings = DB::table('my_bookings')
        ->join('properties', 'properties.id', '=', 'my_bookings.property_id')
        ->whereIn('my_bookings.property_id', $propertyIds)
        ->where('my_bookings.status', 'active')
        ->where('my_bookings.booking_status', 'approved')
        ->select(
            'my_bookings.id',
            'my_bookings.check_in_time',
            'my_bookings.check_out_time',
            'properties.name as property_name'
        )
        ->get();

    // 3. Convert to FullCalendar format
    $events = [];

    foreach ($bookings as $booking) {
        $events[] = [
            'id'    => $booking->id,
            'title' => $booking->property_name .
                       ' (' .
                       date('H:i', strtotime($booking->check_in_time)) .
                       ' - ' .
                       date('H:i', strtotime($booking->check_out_time)) .
                       ')',
            'start' => $booking->check_in_time,
            'end'   => $booking->check_out_time,
        ];
    }

    return response()->json($events);
}

    // Show add city form
    public function create()
    {
        if (!session()->has('admin_id')) {
            return redirect()->route('admin.login');
        }

        $states = State::fetchAllStates();

        $pageHead = View::make('template.page_head')->render();
        $sidebar  = View::make('template.sidebar')->render();
        $pageFoot = View::make('template.page_foot')->render();
        $form = View::make('cities.citiesform', compact('states'))->render();

        return response($pageHead . $sidebar . $form . $pageFoot);
    }
   public function login()
    {
        // if (!session()->has('admin_id')) {
        //     return redirect()->route('admin.login');
        // }


        

        $table = View::make('merchant_panel.login')->render();

        return response( $table);
    }
    public function loginStore(Request $request)
    {
        // validate input
        $request->validate([
            'email'    => 'required|email',
            'password' => 'required|string|min:8',
        ]);

        // call SINGLE model function
        $merchant = Merchant::login(
            $request->email,
            $request->password
        );

        if (!$merchant) {
            return back()
                ->withErrors(['email' => 'Invalid email or password'])
                ->withInput();
        }

        // check merchant status
        if ($merchant->status !== 'active') {
            return back()
                ->withErrors(['email' => 'Your account is inactive. Contact admin.']);
        }

        // create session
        session([
            'merchant_id'    => $merchant->user_id,
            'merchant_name'  => $merchant->name,
            'merchant_email' => $merchant->email,
            'user_type'      => 'merchant',
        ]);
// print_r($session);
// die('lkf');
        // redirect after login
        return redirect()->route('merchant.index');
    }

    // logout
    public function logout()
    {
        session()->forget([
            'merchant_id',
            'merchant_name',
            'merchant_email',
            'user_type',
        ]);

        return redirect()->route('merchant.login');
    }
   public function signup()
    {
        // if (!session()->has('admin_id')) {
        //     return redirect()->route('admin.login');
        // }


     

        $table = View::make('merchant_panel.signup')->render();

        return response($table);
    }
public function signupStore(Request $request)
    {
        // VALIDATION
        $request->validate([
            'name'     => 'required|string|max:255',
            'email'    => 'required|email|max:255',
            'phone_no' => 'nullable|string|max:50',
            'password' => 'required|string|min:8|confirmed',
        ]);

        // check unique email
        if (Merchant::emailExists($request->email)) {
            return back()
                ->withErrors(['email' => 'Email already exists'])
                ->withInput();
        }

        // create merchant
        Merchant::createMerchant([
            'name'     => $request->name,
            'email'    => $request->email,
            'phone_no' => $request->phone_no,
            'password' => $request->password,
        ]);

        return redirect()
            ->route('merchant.login')
            ->with('success', 'Merchant account created successfully. Please login.');
    }
  public function merchant_profile()
    {
        if (!session()->has('merchant_id')) {
            return redirect()->route('merchant.login');
        }

        $merchant = [
            'user_id'   => session('merchant_id'),
            'name'      => session('merchant_name'),
            'email'     => session('merchant_email'),
            'user_type' => session('user_type'),
        ];

        // fetch full user row by user_id (primary key in your schema)
        $user = DB::table('users')->where('user_id', $merchant['user_id'])->first();

        $pageHead = View::make('template.page_head_login',compact('merchant'))->render();
        $sidebar  = View::make('template.page_login_sidebar', compact('merchant'))->render();
        $pageFoot = View::make('template.page_foot_login')->render();

        $table = View::make('merchant_panel.merchant_profile', compact('merchant','user'))->render();

        return response($pageHead . $sidebar . $table . $pageFoot);
    }

    /**
     * Handle profile update
     * - current_password is required for any update (confirmation)
     * - new password optional (min 8 + confirmed)
     */
  public function profileUpdate(Request $request)
{
    if (!session()->has('merchant_id')) {
        return redirect()->route('merchant.login');
    }

    $userId = session('merchant_id'); // users.user_id

    // VALIDATION
    $request->validate([
        'name'     => 'required|string|max:255',
        'email'    => ['required','email','max:255', Rule::unique('users','email')->ignore($userId, 'user_id')],
        'phone_no' => 'nullable|string|max:50',
        'password' => 'nullable|string|min:8|confirmed',
        'profile_image' => 'nullable|image|max:2048',
        'current_password' => 'required|string',
    ]);

    // VERIFY CURRENT PASSWORD
    $dbPassword = DB::table('users')
        ->where('user_id', $userId)
        ->value('password');

    if (!$dbPassword || !Hash::check($request->current_password, $dbPassword)) {
        return back()
            ->withErrors(['current_password' => 'Wrong current password'])
            ->withInput();
    }

    // PRESERVE USER TYPE
    $currentUserType = DB::table('users')
        ->where('user_id', $userId)
        ->value('user_type');

    // UPDATE DATA
    $update = [
        'name'       => $request->name,
        'email'      => $request->email,
        'phone_no'   => $request->phone_no ?: null,
        'user_type'  => $currentUserType,
        'updated_at' => now(),
    ];

    // OPTIONAL PASSWORD UPDATE
    if ($request->filled('password')) {
        $update['password'] = Hash::make($request->password);
    }

    // PROFILE IMAGE (STORE ONLY FILENAME)
    if ($request->hasFile('profile_image') && $request->file('profile_image')->isValid()) {

        // generate filename
        $filename = time() . '_' . uniqid() . '.' .
            $request->file('profile_image')->getClientOriginalExtension();

        // store file
        $request->file('profile_image')
            ->storeAs('profile_images', $filename, 'public');

        // delete old image
        $oldImage = DB::table('users')
            ->where('user_id', $userId)
            ->value('profile_image');

        if ($oldImage && Storage::disk('public')->exists('profile_images/' . $oldImage)) {
            Storage::disk('public')->delete('profile_images/' . $oldImage);
        }

        // save ONLY filename in DB
        $update['profile_image'] = $filename;
    }

    // UPDATE USER
    DB::table('users')
        ->where('user_id', $userId)
        ->update($update);

    // UPDATE SESSION
    session([
        'merchant_name'  => $update['name'],
        'merchant_email' => $update['email'],
        'user_type'      => $currentUserType,
    ]);

    return back()->with('success', 'Profile updated successfully.');
}

public function propertyCreate()
{
    if (!session()->has('merchant_id')) {
        return redirect()->route('merchant.login');
    }

    $merchant = [
        'user_id'   => session('merchant_id'),
        'name'      => session('merchant_name'),
        'email'     => session('merchant_email'),
        'user_type' => session('user_type'),
    ];

    // fetch data
    $states = DB::table('states')->select('id','name')->where('status','active')->orderBy('name')->get();
    $extraServices = DB::table('extra_services')->select('id','name','price','price_type')->where('status','active')->orderBy('name')->get();

    // render page (same pattern as other merchant methods)
    $pageHead = View::make('template.page_head_login', compact('merchant'))->render();
    $sidebar  = View::make('template.page_login_sidebar', compact('merchant'))->render();
    $pageFoot = View::make('template.page_foot_login')->render();

    // property is null for create — same blade used for create+edit
    $property = null;
    $form = View::make('merchant_panel.property_form', compact('merchant','states','extraServices','property'))->render();

    return response($pageHead . $sidebar . $form . $pageFoot);
}

/**
 * Show edit form (pre-fills form)
 */
public function propertyEdit($slug)
{
    if (!session()->has('merchant_id')) {
        return redirect()->route('merchant.login');
    }

    $merchant = [
        'user_id'   => session('merchant_id'),
        'name'      => session('merchant_name'),
        'email'     => session('merchant_email'),
        'user_type' => session('user_type'),
    ];

    $merchantId = $merchant['user_id'];

    // resolve slug -> property (ensures it belongs to this merchant)
    $property = $this->resolveSlugToProperty($slug, $merchantId);

    if (!$property) {
        return redirect()->route('merchant.index')->with('error', 'Property not found or access denied.');
    }

    // load extra resources
    $states = DB::table('states')->select('id','name')->where('status','active')->orderBy('name')->get();
    $extraServices = DB::table('extra_services')->select('id','name','price','price_type')->where('status','active')->orderBy('name')->get();

    $pageHead = View::make('template.page_head_login', compact('merchant'))->render();
    $sidebar  = View::make('template.page_login_sidebar', compact('merchant'))->render();
    $pageFoot = View::make('template.page_foot_login')->render();

    $form = View::make('merchant_panel.property_form', compact('merchant','states','extraServices','property'))->render();

    return response($pageHead . $sidebar . $form . $pageFoot);
}


/**
 * Store (create) property
 */
public function propertyStore(Request $request)
{
    if (!session()->has('merchant_id')) {
        return redirect()->route('merchant.login');
    }

    $merchantId = session('merchant_id');

    $rules = [
        'map_iframe' => 'nullable|string|max:2000',
        'state_id' => 'required|integer|exists:states,id',
        'city_id' => 'required|integer|exists:cities,id',
        'locality_id' => 'required|integer|exists:localities,id',
        'name' => 'required|string|max:255',
        'address' => 'required|string|max:1000',
        'price' => 'required|numeric|min:0',
        'price_unit' => ['required', Rule::in(['per_hour','per_day','one_time'])],
        'features' => 'nullable|string|max:2000',
        'extra_services' => 'nullable|array',
        'extra_services.*' => 'integer|exists:extra_services,id',

        // REQUIRED: at least 5 images when creating
        'images' => 'required|array|min:5|max:10',
        'images.*' => 'image|max:5120', // 5MB each
    ];

    $messages = [
        'images.required' => 'Please upload at least 5 images.',
        'images.min' => 'Minimum 5 images are required to submit a property.',
        'images.max' => 'Maximum 10 images are allowed.',
    ];

    $validator = Validator::make($request->all(), $rules, $messages);

    if ($validator->fails()) {
        return back()->withErrors($validator)->withInput();
    }

    $now = Carbon::now();

    $propData = [
        'admin_id' => null,
        'user_id' => $merchantId,
        'name' => $request->name,
        'state_id' => $request->state_id,
        'city_id' => $request->city_id,
        'locality_id' => $request->locality_id,
        'address' => $request->address,
        'price' => $request->price,
        'price_unit' => $request->price_unit,
        'features' => $request->features ? trim($request->features) : null,
        'extra_services' => $request->extra_services ? implode(',', $request->extra_services) : null,
        'status' => 'pending',
        'map_iframe' => $request->map_iframe ? trim($request->map_iframe) : null,
        'created_at' => $now,
        'updated_at' => $now,
    ];

    DB::beginTransaction();
    try {
        $propertyId = DB::table('properties')->insertGetId($propData);

        // images handling - required and at least 5 already validated
        if ($request->hasFile('images')) {
            foreach ($request->file('images') as $file) {
                if ($file->isValid()) {
                    $filename = time() . '_' . Str::random(8) . '.' . $file->getClientOriginalExtension();
                    // store in public disk under property_images
                    $file->storeAs('property_images', $filename, 'public');

                    DB::table('property_images')->insert([
                        'property_id' => $propertyId,
                        'filename' => $filename,
                        'created_at' => $now,
                        'updated_at' => $now,
                    ]);
                }
            }
        }

        DB::commit();

    } catch (\Exception $e) {
        DB::rollBack();
        \Log::error('propertyStore error: ' . $e->getMessage());
        return back()->withErrors(['error' => 'Failed to create property: ' . $e->getMessage()])->withInput();
    }

    // Build visible slug if you want to redirect to edit immediately (optional)
    // $visibleSlug = Str::slug($request->name ?? 'property-' . $propertyId) . '-' . $propertyId;
    return redirect()->route('merchant.my.properties')->with('success', 'Property created successfully.');
    // OR redirect to edit page:
    // return redirect()->route('merchant.properties.edit', $visibleSlug)->with('success', 'Property created successfully.');
}

/**
 * Update existing property
 * - only merchant owner can update (checked)
 * - appends new uploaded images (does not remove existing images; you can extend to delete)
 */
public function propertyUpdate(Request $request, $slug)
{
    if (!session()->has('merchant_id')) {
        return redirect()->route('merchant.login');
    }

    $merchantId = session('merchant_id');

    // resolve slug -> property (ensures it belongs to this merchant)
    $property = $this->resolveSlugToProperty($slug, $merchantId);
    if (!$property) {
        return redirect()->route('merchant.index')->with('error', 'Property not found or access denied.');
    }

    $id = $property->id; // use numeric id for DB ops

    // COUNT existing images
    $existingImageCount = DB::table('property_images')->where('property_id', $id)->count();
    $newImageCount = $request->hasFile('images') ? count($request->file('images')) : 0;
    $totalImages = $existingImageCount + $newImageCount;

    // Enforce minimum 5 images total on update
    if ($totalImages < 5) {
        return back()
            ->withErrors(['images' => 'At least 5 images are required. You currently have ' . $existingImageCount . ' existing image(s).'])
            ->withInput();
    }

    $rules = [
        'map_iframe' => 'nullable|string|max:2000',
        'state_id' => 'required|integer|exists:states,id',
        'city_id' => 'required|integer|exists:cities,id',
        'locality_id' => 'required|integer|exists:localities,id',
        'name' => 'required|string|max:255',
        'address' => 'required|string|max:1000',
        'price' => 'required|numeric|min:0',
        'price_unit' => ['required', Rule::in(['per_hour','per_day','one_time'])],
        'features' => 'nullable|string|max:2000',
        'extra_services' => 'nullable|array',
        'extra_services.*' => 'integer|exists:extra_services,id',

        // images optional on edit but limited to max 10 at once; overall total checked above
        'images' => 'nullable|array|max:10',
        'images.*' => 'image|max:5120',
    ];

    $validator = Validator::make($request->all(), $rules, [
        'images.max' => 'You can upload up to 10 images at once.',
    ]);

    if ($validator->fails()) {
        return back()->withErrors($validator)->withInput();
    }

    $now = Carbon::now();

    $updateData = [
        'name' => $request->name,
        'state_id' => $request->state_id,
        'city_id' => $request->city_id,
        'locality_id' => $request->locality_id,
        'address' => $request->address,
        'price' => $request->price,
        'price_unit' => $request->price_unit,
        'features' => $request->features ? trim($request->features) : null,
        'extra_services' => $request->extra_services ? implode(',', $request->extra_services) : null,
        'updated_at' => $now,
        'status' => 'pending',
        'map_iframe' => $request->map_iframe ? trim($request->map_iframe) : null,
    ];

    DB::beginTransaction();
    try {
        DB::table('properties')->where('id', $id)->update($updateData);

        // append any new images uploaded
        if ($request->hasFile('images')) {
            foreach ($request->file('images') as $file) {
                if ($file->isValid()) {
                    $filename = time() . '_' . Str::random(8) . '.' . $file->getClientOriginalExtension();
                    $file->storeAs('property_images', $filename, 'public');

                    DB::table('property_images')->insert([
                        'property_id' => $id,
                        'filename' => $filename,
                        'created_at' => $now,
                        'updated_at' => $now,
                    ]);
                }
            }
        }

        DB::commit();
    } catch (\Exception $e) {
        DB::rollBack();
        \Log::error('propertyUpdate error: ' . $e->getMessage());
        return back()->withErrors(['error' => 'Failed to update property: ' . $e->getMessage()])->withInput();
    }

    // Build visible slug for redirects if needed
    // $visibleSlug = Str::slug($request->name ?? $property->name) . '-' . $id;

    return redirect()->route('merchant.my.properties')->with('success', 'Property updated successfully.');
    // OR redirect to edit page using slug:
    // return redirect()->route('merchant.properties.edit', $visibleSlug)->with('success', 'Property updated successfully.');
}

private function resolveSlugToProperty(string $slug, $merchantId = null)
{
    // 1) Try id suffix (fast & reliable): last segment numeric
    $parts = explode('-', $slug);
    $last = end($parts);

    if (ctype_digit($last)) {
        $id = (int) $last;

        $q = DB::table('properties')->where('id', $id);
        if ($merchantId) {
            $q->where('user_id', $merchantId);
        }
        $prop = $q->first();
        if ($prop) {
            return $prop;
        }
        // fall through to name-match fallback
    }

    // 2) Fallback: match slugified name
    // If merchantId provided, restrict to merchant's properties to avoid ambiguity
    $candidatesQ = DB::table('properties')->select('id', 'name');
    if ($merchantId) {
        $candidatesQ->where('user_id', $merchantId);
    }
    $candidates = $candidatesQ->get();

    foreach ($candidates as $c) {
        if (Str::slug($c->name) === $slug) {
            // load full property
            return DB::table('properties')->where('id', $c->id)->first();
        }
    }

    return null;
}
/**
 * AJAX: cities - returns all active cities, or cities for a given state_id
 */
public function ajaxCities(Request $request)
{
    $stateId = $request->query('state_id');

    $q = DB::table('cities')->select('id','name','state_id')->orderBy('name')->where('status','active');
    if ($stateId) $q->where('state_id', $stateId);

    $cities = $q->get();

    return response()->json($cities);
}

/**
 * AJAX: localities - returns all active localities, or localities for a given city_id
 */
public function ajaxLocalities(Request $request)
{
    $cityId = $request->query('city_id');

    $q = DB::table('localities')->select('id','name','city_id')->orderBy('name')->where('status','active');
    if ($cityId) $q->where('city_id', $cityId);

    $localities = $q->get();

    return response()->json($localities);
}
/**
 * Show logged-in merchant's properties list (My Properties)
 */
public function myProperties()
{
    if (!session()->has('merchant_id')) {
        return redirect()->route('merchant.login');
    }

    $merchant = [
        'user_id'   => session('merchant_id'),
        'name'      => session('merchant_name'),
        'email'     => session('merchant_email'),
        'user_type' => session('user_type'),
    ];

    $merchantId = $merchant['user_id'];

    /*
     | Fetch properties with first image via a safe subquery.
     | This avoids GROUP BY issues in strict SQL modes.
     */
    $properties = DB::table('properties as p')
        ->select(
            'p.*',
            DB::raw('(SELECT filename FROM property_images WHERE property_images.property_id = p.id ORDER BY id ASC LIMIT 1) as first_image')
        )
        ->where('p.user_id', $merchantId)
        ->orderByDesc('p.created_at')
        ->get();

    // page layout (same as rest of merchant panel)
    $pageHead = View::make('template.page_head_login', compact('merchant'))->render();
    $sidebar  = View::make('template.page_login_sidebar', compact('merchant'))->render();
    $pageFoot = View::make('template.page_foot_login')->render();

    // IMPORTANT: pass as $ads to reuse the SAME design
    $table = View::make('merchant_panel.my_properties', [
        'merchant' => $merchant,
        'ads' => $properties, // reuse variable name from design
    ])->render();

    return response($pageHead . $sidebar . $table . $pageFoot);
}

public function propertyBookings(Request $request)
{
    if (!session()->has('merchant_id')) {
        return redirect()->route('merchant.login');
    }

    $merchantId = session('merchant_id');

    /*
     STEP 1: Merchant active property IDs
    */
    $myPropertyIds = DB::table('properties')
        ->where(function ($query) use ($merchantId) {
            $query->where('admin_id', $merchantId)
                  ->orWhere('user_id', $merchantId);
        })
        ->where('status', 'active')
        ->pluck('id')
        ->toArray();

    /*
     STEP 2: Base bookings query
    */
    $query = DB::table('my_bookings as b')
        ->join('properties as p', 'p.id', '=', 'b.property_id')
        ->join('users as u', 'u.user_id', '=', 'b.user_id')
        ->leftJoin('states as s', 's.id', '=', 'b.state_id')
        ->leftJoin('cities as c', 'c.id', '=', 'b.city_id')
        ->leftJoin('localities as l', 'l.id', '=', 'b.locality_id')
        ->whereIn('b.property_id', $myPropertyIds)
        ->select([
            'b.id',
            'b.booking_status',
            'b.payment_status',
            'b.payment_method',
            'b.check_in_time',
            'b.check_out_time',
            'b.created_at',

            'p.name as property_name',
            'p.address',
            'p.price',
            'p.price_unit',

            'u.name as user_name',
            'u.email as user_email',

            's.name as state_name',
            'c.name as city_name',
            'l.name as locality_name',
        ])
        ->selectRaw('TIMESTAMPDIFF(HOUR, b.check_in_time, b.check_out_time) as hours')
        ->selectRaw('
            CASE
                WHEN p.price_unit = "per_hour" THEN p.price * TIMESTAMPDIFF(HOUR, b.check_in_time, b.check_out_time)
                WHEN p.price_unit = "per_day" THEN p.price * (TIMESTAMPDIFF(HOUR, b.check_in_time, b.check_out_time) / 24)
                ELSE p.price
            END as amount
        ');

    /*
     STEP 3: ✅ DATE FILTER (CORRECT BOOKING LOGIC)
     Booking overlaps selected day
    */
    if ($request->filled('date')) {
        try {
            $date = Carbon::parse($request->date);
            $start = $date->copy()->startOfDay();
            $end   = $date->copy()->endOfDay();

            $query->where(function ($q) use ($start, $end) {
                $q->whereBetween('b.check_in_time', [$start, $end])
                  ->orWhereBetween('b.check_out_time', [$start, $end])
                  ->orWhere(function ($q2) use ($start, $end) {
                      $q2->where('b.check_in_time', '<=', $start)
                         ->where('b.check_out_time', '>=', $end);
                  });
            });
        } catch (\Exception $e) {
            // ignore invalid date
        }
    }

    $bookings = empty($myPropertyIds)
        ? DB::table('my_bookings')->whereRaw('1 = 0')->paginate(20)
        : $query->orderBy('b.created_at', 'desc')->paginate(20)->withQueryString();

    $merchant = [
        'user_id' => $merchantId,
        'name'    => session('merchant_name'),
        'email'   => session('merchant_email'),
    ];

    $pageHead = View::make('template.page_head_login', compact('merchant'))->render();
    $sidebar  = View::make('template.page_login_sidebar', compact('merchant'))->render();
    $content  = View::make('merchant_panel.my_bookings', compact('bookings', 'merchant'))->render();
    $pageFoot = View::make('template.page_foot_login')->render();

    return response($pageHead . $sidebar . $content . $pageFoot);
}

public function kycForm()
{
    if (!session()->has('merchant_id')) {
        return redirect()->route('merchant.login');
    }

    $merchant = [
        'user_id'   => session('merchant_id'),
        'name'      => session('merchant_name'),
        'email'     => session('merchant_email'),
        'user_type' => session('user_type'),
    ];

    $merchantId = $merchant['user_id'];

    // fetch latest KYC
    $kyc = DB::table('merchant_kyc')
        ->where('user_id', $merchantId)
        ->orderByDesc('id')
        ->first();

    /**
     * STATUS RULES
     */
    // 1️⃣ Approved → redirect to dashboard
    // if ($kyc && $kyc->status === 'approved') {
    //     return redirect()->route('merchant.index')
    //         ->with('success', 'Your KYC is already approved.');
    // }

    // layout
    $pageHead = View::make('template.page_head_login', compact('merchant'))->render();
    $sidebar  = View::make('template.page_login_sidebar', compact('merchant'))->render();
    $pageFoot = View::make('template.page_foot_login')->render();

    // 2️⃣ Pending → show read-only view
    if ($kyc && $kyc->status === 'pending') {
        $content = View::make('merchant_panel.kyc_form', [
            'merchant' => $merchant,
            'kyc'      => $kyc,
        ])->render();

        return response($pageHead . $sidebar . $content . $pageFoot);
    }

    // 3️⃣ No KYC OR Rejected → show editable form
    $content = View::make('merchant_panel.kyc_form', [
        'merchant' => $merchant,
        'kyc'      => $kyc, // pass for rejected case
    ])->render();

    return response($pageHead . $sidebar . $content . $pageFoot);
}

public function kycStore(Request $request)
{
    $merchantId = session('merchant_id');

    // fetch latest kyc
    $existingKyc = DB::table('merchant_kyc')
        ->where('user_id', $merchantId)
        ->orderByDesc('id')
        ->first();

    /**
     * 🚫 BLOCK CONDITIONS
     */
    if ($existingKyc && in_array($existingKyc->status, ['pending', 'approved'])) {
        return redirect()->route('merchant.index')
            ->with('error', 'You cannot update KYC while it is pending or approved.');
    }

    /**
     * ✅ VALIDATION
     */
    $request->validate([
        'document_type' => 'required|in:aadhaar,other',
        'name' => 'required|string|max:255',
        'dob' => 'required|date',
        'address' => 'required|string',
        'front_image' => 'required|image|max:5120',
        'back_image' => 'nullable|image|max:5120',
    ]);

    /**
     * ✅ FILE UPLOADS
     */
    $frontFile = $request->file('front_image');
    $frontName = uniqid().'_'.$frontFile->getClientOriginalName();
    $frontFile->storeAs('kyc', $frontName, 'public');

    $backName = null;
    if ($request->document_type === 'aadhaar' && $request->hasFile('back_image')) {
        $backFile = $request->file('back_image');
        $backName = uniqid().'_'.$backFile->getClientOriginalName();
        $backFile->storeAs('kyc', $backName, 'public');
    }

    /**
     * 🔁 INSERT OR UPDATE
     */
    if ($existingKyc && $existingKyc->status === 'rejected') {
        // UPDATE rejected KYC
        DB::table('merchant_kyc')
            ->where('id', $existingKyc->id)
            ->update([
                'document_type' => $request->document_type,
                'name' => $request->name,
                'dob' => $request->dob,
                'address' => $request->address,
                'front_image' => $frontName,
                'back_image' => $backName,
                'status' => 'pending', // 🔁 goes back to review
                'updated_at' => now(),
            ]);
    } else {
        // INSERT first-time KYC
        DB::table('merchant_kyc')->insert([
            'user_id' => $merchantId,
            'document_type' => $request->document_type,
            'name' => $request->name,
            'dob' => $request->dob,
            'address' => $request->address,
            'front_image' => $frontName,
            'back_image' => $backName,
            'status' => 'pending',
            'created_at' => now(),
            'updated_at' => now(),
        ]);
    }

    return redirect()->route('merchant.kyc.form')
        ->with('success', 'KYC submitted successfully. Status: Pending');
}
public function bankForm()
{
    if (!session()->has('merchant_id')) {
        return redirect()->route('merchant.login');
    }

    $merchant = [
        'user_id'   => session('merchant_id'),
        'name'      => session('merchant_name'),
        'email'     => session('merchant_email'),
        'user_type' => session('user_type'),
    ];

    $accounts = DB::table('merchant_bank_accounts')
        ->where('user_id', $merchant['user_id'])
        ->orderByDesc('id')
        ->get();

    $pageHead = View::make('template.page_head_login', compact('merchant'))->render();
    $sidebar  = View::make('template.page_login_sidebar', compact('merchant'))->render();
    $pageFoot = View::make('template.page_foot_login')->render();

    $content = View::make('merchant_panel.bank_details', [
        'merchant' => $merchant,
        'accounts' => $accounts,
    ])->render();

    return response($pageHead . $sidebar . $content . $pageFoot);
}
public function bankStore(Request $request)
{
    $merchantId = session('merchant_id');

    $request->validate([
        'name_in_bank' => 'required|string|max:255',
        'bank_name' => 'required|string|max:255',
        'branch_name' => 'nullable|string|max:255',
        'ifsc' => 'required|string|max:32',
        'account_number' => 'required|string|max:64',
        'confirm_account_number' => 'required|same:account_number',
    ]);

    DB::table('merchant_bank_accounts')->insert([
        'user_id' => $merchantId,
        'name_in_bank' => $request->name_in_bank,
        'bank_name' => $request->bank_name,
        'branch_name' => $request->branch_name,
        'ifsc' => $request->ifsc,
        'account_number' => $request->account_number,
        'status' => 'pending',
        'created_at' => now(),
        'updated_at' => now(),
    ]);

    return redirect()->route('merchant.bank.form')
        ->with('success', 'Bank account submitted. Status: Pending approval.');
}
public function bankUpdate(Request $request)
{
    $merchantId = session('merchant_id');

    $request->validate([
        'account_id' => 'required|integer',
        'name_in_bank' => 'required|string|max:255',
        'bank_name' => 'required|string|max:255',
        'branch_name' => 'nullable|string|max:255',
        'ifsc' => 'required|string|max:32',
        'account_number' => 'required|string|max:64',
        'confirm_account_number' => 'required|same:account_number',
    ]);

    $account = DB::table('merchant_bank_accounts')
        ->where('id', $request->account_id)
        ->where('user_id', $merchantId)
        ->first();

    if (!$account) {
        return redirect()->route('merchant.bank.form')
            ->with('error', 'Account not found.');
    }

    DB::table('merchant_bank_accounts')
        ->where('id', $account->id)
        ->update([
            'name_in_bank' => $request->name_in_bank,
            'bank_name' => $request->bank_name,
            'branch_name' => $request->branch_name,
            'ifsc' => $request->ifsc,
            'account_number' => $request->account_number,
            'status' => 'pending',
            'updated_at' => now(),
        ]);

    return redirect()->route('merchant.bank.form')
        ->with('success', 'Bank account updated. Status: Pending approval.');
}
public function withdrawalForm()
{
    if (!session()->has('merchant_id')) {
        return redirect()->route('merchant.login');
    }

    $merchant = [
        'user_id'   => session('merchant_id'),
        'name'      => session('merchant_name'),
        'email'     => session('merchant_email'),
        'user_type' => session('user_type'),
    ];

    $userId = $merchant['user_id'];

    // --- KYC check: latest entry must be approved
    $kyc = DB::table('merchant_kyc')
        ->where('user_id', $userId)
        ->orderByDesc('id')
        ->first();

    if (!$kyc || $kyc->status !== 'approved') {
        return redirect()->route('merchant.kyc.form')
            ->with('error', 'You must complete KYC (approved) before making withdrawals.');
    }

    // fetch active bank accounts for this merchant (only status = active)
    $banks = DB::table('merchant_bank_accounts')
        ->where('user_id', $userId)
        ->where('status', 'active')
        ->orderByDesc('id')
        ->get();

    if ($banks->isEmpty()) {
        return redirect()->route('merchant.bank.form')
            ->with('error', 'Please add and verify an active bank account before making a withdrawal.');
    }

    // fetch wallet balance
    $wallet = DB::table('user_wallet')->where('user_id', $userId)->first();
    $balance = $wallet ? (float)$wallet->balance : 0.00;

    // Minimum balance required to be eligible to request a withdrawal
    $minWithdrawalBalance = 1000.00;
    $canWithdraw = ($balance >= $minWithdrawalBalance);

    $pageHead = View::make('template.page_head_login', compact('merchant'))->render();
    $sidebar  = View::make('template.page_login_sidebar', compact('merchant'))->render();
    $pageFoot = View::make('template.page_foot_login')->render();

    $content = View::make('merchant_panel.withdrawal_form', [
        'merchant' => $merchant,
        'banks' => $banks,
        'balance' => $balance,
        'canWithdraw' => $canWithdraw,
        'minWithdrawalBalance' => $minWithdrawalBalance,
    ])->render();

    return response($pageHead . $sidebar . $content . $pageFoot);
}

public function withdrawalStore(Request $request)
{
    if (!session()->has('merchant_id')) {
        return redirect()->route('merchant.login');
    }

    $userId = session('merchant_id');

    // Validate
    $request->validate([
        'bank_account_id' => 'required|integer',
        'amount' => 'required|numeric|min:1',
    ]);

    $amount = (float)$request->input('amount');
    $bankAccountId = (int)$request->input('bank_account_id');

    // Ensure the bank belongs to user and is active
    $bank = DB::table('merchant_bank_accounts')
        ->where('id', $bankAccountId)
        ->where('user_id', $userId)
        ->where('status', 'active')
        ->first();

    if (!$bank) {
        return redirect()->back()->with('error', 'Selected bank account not found or not active.');
    }

    // fetch wallet and check balance
    $wallet = DB::table('user_wallet')->where('user_id', $userId)->lockForUpdate()->first();

    $balance = $wallet ? (float)$wallet->balance : 0.00;
    if ($balance < $amount) {
        return redirect()->back()->with('error', 'Insufficient wallet balance to make this withdrawal.');
    }

    // Use DB transaction for atomicity
    DB::beginTransaction();
    try {
        // deduct wallet
        DB::table('user_wallet')
            ->where('user_id', $userId)
            ->update([
                'balance' => DB::raw("GREATEST(balance - " . (float)$amount . ", 0)"),
                'updated_at' => now(),
            ]);

        // insert withdrawal record (status = pending)
        $withdrawalId = DB::table('withdrawals')->insertGetId([
            'user_id' => $userId,
            'bank_account_id' => $bankAccountId,
            'amount' => $amount,
            'status' => 'pending',
            'admin_note' => null,
            'created_at' => now(),
            'updated_at' => now(),
        ]);

        // insert transaction record (note links to withdrawal)
        DB::table('transactions')->insert([
            'user_id' => $userId,
            'hideout_id' => null,
            'status' => 'pending',
            'admin_status' => 'pending',
            'admin_note' => null,
            'note' => 'Withdrawal request #W-' . $withdrawalId,
            'amount' => $amount,
            'created_at' => now(),
            'updated_at' => now(),
        ]);

        DB::commit();

        return redirect()->route('merchant.withdraw.history')
            ->with('success', 'Withdrawal request submitted. Status: Pending.');
    } catch (\Exception $e) {
        DB::rollBack();
        // optionally log error
        \Log::error('Withdraw store error: ' . $e->getMessage());
        return redirect()->back()->with('error', 'Unable to process withdrawal right now. Please try again later.');
    }
}

public function withdrawalHistory()
{
    if (!session()->has('merchant_id')) {
        return redirect()->route('merchant.login');
    }

    $merchant = [
        'user_id'   => session('merchant_id'),
        'name'      => session('merchant_name'),
        'email'     => session('merchant_email'),
        'user_type' => session('user_type'),
    ];

    $userId = $merchant['user_id'];

    // fetch withdrawals with bank info
    $withdrawals = DB::table('withdrawals as w')
        ->leftJoin('merchant_bank_accounts as b', 'b.id', '=', 'w.bank_account_id')
        ->select('w.*', 'b.bank_name', 'b.branch_name', 'b.name_in_bank')
        ->where('w.user_id', $userId)
        ->orderByDesc('w.created_at')
        ->get();

    $pageHead = View::make('template.page_head_login', compact('merchant'))->render();
    $sidebar  = View::make('template.page_login_sidebar', compact('merchant'))->render();
    $pageFoot = View::make('template.page_foot_login')->render();

    $content = View::make('merchant_panel.withdrawal_history', [
        'merchant' => $merchant,
        'withdrawals' => $withdrawals,
    ])->render();

    return response($pageHead . $sidebar . $content . $pageFoot);
}

public function softDelete($slug)
{
    if (!session()->has('merchant_id')) {
        return response()->json([
            'success' => false,
            'message' => 'Unauthorized'
        ], 401);
    }

    $merchantId = session('merchant_id');

    // Resolve slug → property
    $property = $this->resolveSlugToProperty($slug, $merchantId);

    if (!$property) {
        return response()->json([
            'success' => false,
            'message' => 'Property not found'
        ], 404);
    }

    // SOFT DELETE = mark inactive
    DB::table('properties')
        ->where('id', $property->id)
        ->update([
            'status' => 'inactive',
            'updated_at' => now(),
        ]);

    return response()->json([
        'success' => true,
        'message' => 'Property marked inactive'
    ]);
}

}
