<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\View;
use App\Models\State;
use Illuminate\Support\Str;
use Illuminate\Support\Facades\Storage;
use DB;

class PropertyController extends Controller
{
    public function index()
    {
        if (!session()->has('admin_id')) {
            return redirect()->route('admin.login');
        }

        $properties = State::fetchAllProperties();

        $pageHead = View::make('template.page_head')->render();
        $sidebar  = View::make('template.sidebar')->render();
        $pageFoot = View::make('template.page_foot')->render();
        $table = View::make('properties.listing', compact('properties'))->render();

        return response($pageHead . $sidebar . $table . $pageFoot);
    }

    public function create()
    {
        if (!session()->has('admin_id')) {
            return redirect()->route('admin.login');
        }

        $states = State::fetchStates();
        $services = State::fetchAllExtraServices();

        $pageHead = View::make('template.page_head')->render();
        $sidebar  = View::make('template.sidebar')->render();
        $pageFoot = View::make('template.page_foot')->render();
        $form = View::make('properties.form', compact('states','services'))->render();

        return response($pageHead . $sidebar . $form . $pageFoot);
    }

    public function store(Request $request)
    {
        $request->validate([
            'name' => 'required|string|max:255',
            'state_id' => 'required|integer|exists:states,id',
            'city_id' => 'required|integer|exists:cities,id',
            'locality_id' => 'nullable|integer|exists:localities,id',
            'address' => 'required|string|max:2000',
            'price' => 'required|numeric|min:0',
            'price_unit' => 'required|in:per_hour,per_day,one_time',
            'latitude' => 'nullable|numeric',
            'longitude' => 'nullable|numeric',
            'images.*' => 'nullable|image|mimes:jpeg,png,jpg,gif|max:5120',
            'images' => 'nullable|array|max:6',
            'extra_services' => 'nullable|array',
        ]);

        DB::beginTransaction();
        try {
            $propertyId = State::createProperty($request->only([
                'name','state_id','city_id','locality_id','address','price','price_unit','latitude','longitude'
            ]));

            if ($request->filled('extra_services')) {
                State::attachServicesToProperty($propertyId, $request->input('extra_services'));
            }

            if ($request->hasFile('images')) {
                $files = $request->file('images');
                $count = 0;
                foreach ($files as $file) {
                    if ($count >= 6) break;
                    $filename = Str::random(12) . '.' . $file->getClientOriginalExtension();
$path = $file->storeAs("public/uploads/properties/{$propertyId}", $filename);
                    State::savePropertyImage($propertyId, $filename);
                    $count++;
                }
            }

            DB::commit();
        } catch (\Exception $e) {
            DB::rollBack();
            return back()->withErrors(['error' => $e->getMessage()])->withInput();
        }

        return redirect()->route('properties.index')->with('success','Property added successfully');
    }

   public function edit($slug)
{
    if (!session()->has('admin_id')) {
        return redirect()->route('admin.login');
    }

    $property = $this->resolveSlugToProperty($slug);
    if (!$property) {
        return redirect()->route('properties.index')->withErrors(['Property not found']);
    }

    $states = State::fetchStates();
    $cities = State::fetchCitiesByState($property->state_id);
    $localities = State::fetchLocalitiesByCity($property->city_id);
    $services = State::fetchAllExtraServices();
    $selectedServices = State::fetchServicesForProperty($property->id);

    $pageHead = View::make('template.page_head')->render();
    $sidebar  = View::make('template.sidebar')->render();
    $pageFoot = View::make('template.page_foot')->render();
    $form = View::make('properties.form', compact('property','states','cities','localities','services','selectedServices'))->render();

    return response($pageHead . $sidebar . $form . $pageFoot);
}

   public function update(Request $request, $slug)
{
    // input validation (keep yours)
    $request->validate([
        'name' => 'required|string|max:255',
        'state_id' => 'required|integer|exists:states,id',
        'city_id' => 'required|integer|exists:cities,id',
        'locality_id' => 'nullable|integer|exists:localities,id',
        'address' => 'required|string|max:2000',
        'price' => 'required|numeric|min:0',
        'price_unit' => 'required|in:per_hour,per_day,one_time',
        'latitude' => 'nullable|numeric',
        'longitude' => 'nullable|numeric',
        'images.*' => 'nullable|image|mimes:jpeg,png,jpg,gif|max:5120',
        'images' => 'nullable|array',
        'extra_services' => 'nullable|array',
    ]);

    $property = $this->resolveSlugToProperty($slug);
    if (!$property) {
        return redirect()->route('properties.index')->withErrors(['Property not found']);
    }

    $id = $property->id;

    DB::beginTransaction();
    try {
        State::updateProperty($id, $request->only([
            'name','state_id','city_id','locality_id','address','price','price_unit','latitude','longitude'
        ]));

        State::syncServicesToProperty($id, $request->input('extra_services', []));

        if ($request->hasFile('images')) {
            $existingCount = State::countPropertyImages($id);
            $files = $request->file('images');
            $allowed = max(0, 6 - $existingCount);
            $files = array_slice($files, 0, $allowed);
            foreach ($files as $file) {
                $filename = Str::random(12) . '.' . $file->getClientOriginalExtension();
                $path = $file->storeAs("public/uploads/properties/{$id}", $filename);
                State::savePropertyImage($id, $filename);
            }
        }

        DB::commit();
    } catch (\Exception $e) {
        DB::rollBack();
        return back()->withErrors(['error' => $e->getMessage()])->withInput();
    }

    return redirect()->route('properties.index')->with('success','Property updated successfully');
}

 public function destroy($slug)
{
    $property = $this->resolveSlugToProperty($slug);
    if (!$property) {
        return redirect()->route('properties.index')->with('error','Property not found');
    }

    State::deactivateProperty($property->id);
    return redirect()->route('properties.index')->with('success','Property deactivated successfully');
}


    // AJAX endpoints for dynamic selects
public function getCities($state_id)
{
    try {
        $stateId = (int)$state_id;
        
        // Add validation
        if ($stateId <= 0) {
            return response()->json([
                'success' => false, 
                'message' => 'Invalid state ID',
                'data' => []
            ], 400);
        }
        
        $cities = State::fetchCitiesByState($stateId);
        
        // Add logging for debugging
        \Log::info('Fetching cities for state: ' . $stateId . ', Found: ' . count($cities));
        
        return response()->json([
            'success' => true, 
            'data' => $cities,
            'message' => count($cities) . ' cities found'
        ]);
        
    } catch (\Exception $e) {
        \Log::error('Error fetching cities: ' . $e->getMessage());
        
        return response()->json([
            'success' => false, 
            'message' => 'Error fetching cities: ' . $e->getMessage(),
            'data' => []
        ], 500);
    }
}

/**
 * Get localities by city ID (AJAX endpoint)  
 */
public function getLocalities($city_id)
{
    try {
        $cityId = (int)$city_id;
        
        // Add validation
        if ($cityId <= 0) {
            return response()->json([
                'success' => false, 
                'message' => 'Invalid city ID',
                'data' => []
            ], 400);
        }
        
        $localities = State::fetchLocalitiesByCity($cityId);
        
        // Add logging for debugging
        \Log::info('Fetching localities for city: ' . $cityId . ', Found: ' . count($localities));
        
        return response()->json([
            'success' => true, 
            'data' => $localities,
            'message' => count($localities) . ' localities found'
        ]);
        
    } catch (\Exception $e) {
        \Log::error('Error fetching localities: ' . $e->getMessage());
        
        return response()->json([
            'success' => false, 
            'message' => 'Error fetching localities: ' . $e->getMessage(), 
            'data' => []
        ], 500);
    }
}
public function approve(Request $request, $slug)
{
    if (!session()->has('admin_id')) {
        return redirect()->route('admin.login');
    }

    $prop = $this->resolveSlugToProperty($slug);
    if (!$prop) {
        return back()->with('error', 'Property not found.');
    }

    if ($prop->status !== 'pending') {
        return back()->with('error', 'Only pending properties can be approved.');
    }

    DB::table('properties')->where('id', $prop->id)->update([
        'status' => 'active',
        'updated_at' => now(),
    ]);

    return back()->with('success', 'Property approved and now active.');
}

public function reject(Request $request, $slug)
{
    if (!session()->has('admin_id')) {
        return redirect()->route('admin.login');
    }

    $prop = $this->resolveSlugToProperty($slug);
    if (!$prop) {
        return back()->with('error', 'Property not found.');
    }

    if ($prop->status !== 'pending') {
        return back()->with('error', 'Only pending properties can be rejected.');
    }

    DB::table('properties')->where('id', $prop->id)->update([
        'status' => 'reject',
        'updated_at' => now(),
    ]);

    return back()->with('success', 'Property rejected.');
}


public function kycIndex()
{
    if (!session()->has('admin_id')) {
        return redirect()->route('admin.login');
    }

    $kycs = DB::table('merchant_kyc as k')
        ->leftJoin('users as u', 'u.user_id', '=', 'k.user_id')
        ->select(
            'k.*',
            'u.name as merchant_name',
            'u.email as merchant_email'
        )
        ->orderByDesc('k.created_at')
        ->get();

    $pageHead = View::make('template.page_head')->render();
    $sidebar  = View::make('template.sidebar')->render();
    $pageFoot = View::make('template.page_foot')->render();

    $table = View::make('properties.merchant_kyc_listing', compact('kycs'))->render();

    return response($pageHead . $sidebar . $table . $pageFoot);
}
public function kycApprove($id)
{
    if (!session()->has('admin_id')) {
        return redirect()->route('admin.login');
    }

    DB::table('merchant_kyc')
        ->where('id', $id)
        ->where('status', 'pending')
        ->update([
            'status' => 'approved',
            'updated_at' => now()
        ]);

    return back()->with('success', 'KYC Approved Successfully.');
}
public function kycReject($id)
{
    if (!session()->has('admin_id')) {
        return redirect()->route('admin.login');
    }

    DB::table('merchant_kyc')
        ->where('id', $id)
        ->where('status', 'pending')
        ->update([
            'status' => 'rejected',
            'updated_at' => now()
        ]);

    return back()->with('success', 'KYC Rejected Successfully.');
}
public function kycView($id)
{
    if (!session()->has('admin_id')) {
        return redirect()->route('admin.login');
    }

    $kyc = DB::table('merchant_kyc as k')
        ->join('users as u', 'u.user_id', '=', 'k.user_id')
        ->select(
            'k.*',
            'u.name as merchant_name',
            'u.email as merchant_email',
            'u.phone_no'
        )
        ->where('k.id', $id)
        ->first();

    if (!$kyc) {
        return redirect()->route('admin.kyc.index')
            ->with('error', 'KYC not found');
    }

    $pageHead = View::make('template.page_head')->render();
    $sidebar  = View::make('template.sidebar')->render();
    $pageFoot = View::make('template.page_foot')->render();

    $content = View::make('properties.merchant_kyc_view', compact('kyc'))->render();

    return response($pageHead . $sidebar . $content . $pageFoot);
}

public function bankAccounts()
{
    if (!session()->has('admin_id')) {
        return redirect()->route('admin.login');
    }

    $banks = DB::table('merchant_bank_accounts as b')
        ->join('users as u', 'u.user_id', '=', 'b.user_id')
        ->where('b.status', 'pending')
        ->select([
            'b.id',
            'b.user_id',
            'b.name_in_bank',
            'b.bank_name',
            'b.ifsc',
            'b.account_number',
            'b.created_at',
            'u.name as merchant_name',
            'u.email as merchant_email',
        ])
        ->orderByDesc('b.created_at')
        ->get();

    $pageHead = View::make('template.page_head')->render();
    $sidebar  = View::make('template.sidebar')->render();
    $pageFoot = View::make('template.page_foot')->render();

    $content = View::make('properties.merchant_bank_accounts_listing', compact('banks'))->render();

    return response($pageHead . $sidebar . $content . $pageFoot);
}
public function approveBankAccount($id)
{
    if (!session()->has('admin_id')) {
        return redirect()->route('admin.login');
    }

    DB::table('merchant_bank_accounts')
        ->where('id', $id)
        ->where('status', 'pending')
        ->update([
            'status' => 'active',
            'updated_at' => now(),
        ]);

    return back()->with('success', 'Bank account approved.');
}
public function rejectBankAccount($id)
{
    if (!session()->has('admin_id')) {
        return redirect()->route('admin.login');
    }

    DB::table('merchant_bank_accounts')
        ->where('id', $id)
        ->where('status', 'pending')
        ->update([
            'status' => 'inactive',
            'updated_at' => now(),
        ]);

    return back()->with('success', 'Bank account rejected.');
}

public function show($slug)
{
    if (!session()->has('admin_id')) {
        return redirect()->route('admin.login');
    }

    $property = $this->resolveSlugToProperty($slug);

    if (!$property) {
        return redirect()->route('properties.index')->with('error', 'Property not found.');
    }

    $images = DB::table('property_images')
        ->where('property_id', $property->id)
        ->orderBy('id')
        ->get();
// echo'<pre>';
// print_r($property);
// die('sdlkf');
    $pageHead = View::make('template.page_head')->render();
    $sidebar  = View::make('template.sidebar')->render();
    $pageFoot = View::make('template.page_foot')->render();
    $view = View::make('properties.property_view', compact('property', 'images'))->render();

    return response($pageHead . $sidebar . $view . $pageFoot);
}
private function resolveSlugToProperty($slug)
{
    // 1) Attempt id-suffix parse: last segment numeric?
    $parts = explode('-', $slug);
    $last = end($parts);
    if (ctype_digit($last)) {
        $id = (int)$last;
        $property = DB::table('properties as p')
            ->leftJoin('states as s', 's.id', '=', 'p.state_id')
            ->leftJoin('cities as c', 'c.id', '=', 'p.city_id')
            ->select('p.*', 's.name as state_name', 'c.name as city_name')
            ->where('p.id', $id)
            ->first();

        if ($property) {
            return $property;
        }
        // if not found by id, continue to fallback
    }

    // 2) Fallback: match slugified name — ONLY if id-suffix not present or didn't match
    // Note: This fetches all properties (lightweight for small datasets but not recommended for very large tables)
    $candidates = DB::table('properties')->select('id', 'name')->get();
    foreach ($candidates as $c) {
        if (Str::slug($c->name) === $slug) {
            // load full property with joins
            return DB::table('properties as p')
                ->leftJoin('states as s', 's.id', '=', 'p.state_id')
                ->leftJoin('cities as c', 'c.id', '=', 'p.city_id')
                ->select('p.*', 's.name as state_name', 'c.name as city_name')
                ->where('p.id', $c->id)
                ->first();
        }
    }

    return null;
}

}
