<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\View;
use App\Models\State; // <-- using your model named 'states'

class StateController extends Controller
{
    // List states
    public function index()
    {
        if (!session()->has('admin_id')) {
            return redirect()->route('admin.login');
        }

        // fetch from your model named 'states'
        $states = State::fetchAllStates();

        $pageHead = View::make('template.page_head')->render();
        $sidebar  = View::make('template.sidebar')->render();
        $pageFoot = View::make('template.page_foot')->render();

        $table = View::make('states.statelisting', compact('states'))->render();

        return response($pageHead . $sidebar . $table . $pageFoot);
    }

    // Show add state form
    public function create()
    {
        if (!session()->has('admin_id')) {
            return redirect()->route('admin.login');
        }

        $pageHead = View::make('template.page_head')->render();
        $sidebar  = View::make('template.sidebar')->render();
        $pageFoot = View::make('template.page_foot')->render();
        $form = View::make('states.stateform')->render();

        return response($pageHead . $sidebar . $form . $pageFoot);
    }

    // Store new state
    public function store(Request $request)
    {
        $request->validate([
            'name'   => 'required|string|max:255',
            'status' => 'required|in:active,inactive',
        ]);

        State::createState($request->only(['name','status']));

        return redirect()->route('states.index')->with('success','State created successfully');
    }

    // Show edit state form
    public function edit($id)
    {
        if (!session()->has('admin_id')) {
            return redirect()->route('admin.login');
        }

        $state = State::fetchStateById($id);

        $pageHead = View::make('template.page_head')->render();
        $sidebar  = View::make('template.sidebar')->render();
        $pageFoot = View::make('template.page_foot')->render();
        $form = View::make('states.stateform', compact('state'))->render();

        return response($pageHead . $sidebar . $form . $pageFoot);
    }

    // Update state
    public function update(Request $request, $id)
    {
        $request->validate([
            'name'   => 'required|string|max:255',
            'status' => 'required|in:active,inactive',
        ]);

        State::updateState($id, $request->only(['name','status']));

        return redirect()->route('states.index')->with('success','State updated successfully');
    }

    // Delete (soft-like) state - set to inactive
    public function destroy($id)
    {
        $state = State::fetchStateById($id);
        if ($state) {
            $state->update(['status' => 'inactive']);
        }

        return redirect()->route('states.index')->with('success', 'State deactivated successfully');
    }
}
