<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\View;
use App\Models\User;
use Illuminate\Support\Facades\DB;
class UserController extends Controller
{
    // List users
    public function index(Request $request)
    {
        if (!session()->has('admin_id')) {
            return redirect()->route('admin.login');
        }

        // read user_type from query string: ?user_type=user or ?user_type=merchant
        $user_type = $request->query('user_type');

        $users = User::fetchAllUsers($user_type);

        $pageHead = View::make('template.page_head')->render();
        $sidebar  = View::make('template.sidebar')->render();
        $pageFoot = View::make('template.page_foot')->render();

        // pass user_type too so view can highlight selected filter
        $table = View::make('users.index', compact('users', 'user_type'))->render();

        return response($pageHead . $sidebar . $table . $pageFoot);
    }

    // Show add user form
    public function create()
    {
        if (!session()->has('admin_id')) {
            return redirect()->route('admin.login');
        }

        $pageHead = View::make('template.page_head')->render();
        $sidebar  = View::make('template.sidebar')->render();
        $pageFoot = View::make('template.page_foot')->render();
        $form = View::make('users.form')->render();

        return response($pageHead . $sidebar . $form . $pageFoot);
    }

    // Store new user
    public function store(Request $request)
    {
        $request->validate([
            'name' => 'required|string|max:255',
            'email' => 'required|email',
            'user_type' => 'required|in:user,merchant'
        ]);

        User::createUser($request->all());
        return redirect()->route('users.index')->with('success','User created successfully');
    }

    // Show edit user form
    public function edit($id)
    {
        if (!session()->has('admin_id')) {
            return redirect()->route('admin.login');
        }

        $user = User::fetchUserById($id);

        $pageHead = View::make('template.page_head')->render();
        $sidebar  = View::make('template.sidebar')->render();
        $pageFoot = View::make('template.page_foot')->render();
        $form = View::make('users.form', compact('user'))->render();

        return response($pageHead . $sidebar . $form . $pageFoot);
    }

    // Update user
    public function update(Request $request, $id)
    {
        $request->validate([
            'name' => 'required|string|max:255',
            'email' => 'required|email',
            'user_type' => 'required|in:user,merchant'
        ]);

        User::updateUser($id, $request->all());
        return redirect()->route('users.index')->with('success','User updated successfully');
    }

    // Delete user (soft)
    public function destroy($id)
    {
        User::where('id', $id)->update(['status' => 'inactive']);
        return redirect()->route('users.index')->with('success', 'User deactivated successfully');
    }

    // Withdrawals listing
public function withdrawals(Request $request)
{
    if (!session()->has('admin_id')) {
        return redirect()->route('admin.login');
    }

    $status = $request->query('status'); // pending / approved / rejected or null

    $q = DB::table('withdrawals as w')
        // join users on their PK user_id (not id)
        ->leftJoin('users as u', 'u.user_id', '=', 'w.user_id')
        // left join bank account if available
        ->leftJoin('merchant_bank_accounts as b', 'b.id', '=', 'w.bank_account_id')
        ->select([
            'w.*',
            'u.user_id as u_user_id',
            'u.name as user_name',
            'u.email as user_email',
            'b.name_in_bank',
            'b.bank_name',
            'b.branch_name',
            'b.ifsc',
            'b.account_number as bank_account_number',
        ]);

    if ($status && in_array($status, ['pending','approved','rejected'])) {
        $q->where('w.status', $status);
    }

    $withdrawals = $q->orderByDesc('w.id')->get();

    $pageHead = View::make('template.page_head')->render();
    $sidebar  = View::make('template.sidebar')->render();
    $pageFoot = View::make('template.page_foot')->render();

    $table = View::make('users.withdrawals', compact('withdrawals', 'status'))->render();

    return response($pageHead . $sidebar . $table . $pageFoot);
}

/**
 * Approve withdrawal (admin)
 */
public function approveWithdrawal(Request $request, $id)
{
    if (!session()->has('admin_id')) {
        return redirect()->route('admin.login');
    }

    $request->validate([
        'admin_note' => 'required|string',
    ]);

    $withdrawal = DB::table('withdrawals')->where('id', $id)->first();

    if (!$withdrawal || $withdrawal->status !== 'pending') {
        return back()->with('error', 'Invalid withdrawal or not pending.');
    }

    DB::transaction(function () use ($id, $withdrawal, $request) {
        // mark withdrawal approved
        DB::table('withdrawals')->where('id', $id)
            ->update([
                'status' => 'approved',
                'admin_note' => $request->admin_note,
                'updated_at' => now(),
            ]);

        // update matching transaction(s) that reference this withdrawal (by note)
    DB::table('transactions')
    ->where('note', 'like', '%W-' . $withdrawal->id . '%')
    ->update([
        'admin_status' => 'approved',   // ✅ allowed
        'status'       => 'completed',  // ✅ allowed
        'admin_note'   => $request->admin_note,
        'updated_at'   => now(),
    ]);

    });

    return back()->with('success', 'Withdrawal approved.');
}

/**
 * Reject withdrawal (admin) and credit back wallet
 */
public function rejectWithdrawal(Request $request, $id)
{
    if (!session()->has('admin_id')) {
        return redirect()->route('admin.login');
    }

    $request->validate([
        'admin_note' => 'required|string',
    ]);

    $withdrawal = DB::table('withdrawals')->where('id', $id)->first();

    if (!$withdrawal || $withdrawal->status !== 'pending') {
        return back()->with('error', 'Invalid withdrawal or not pending.');
    }

    DB::transaction(function () use ($id, $withdrawal, $request) {
        // 1. mark withdrawal rejected
        DB::table('withdrawals')->where('id', $id)
            ->update([
                'status' => 'rejected',
                'admin_note' => $request->admin_note,
                'updated_at' => now(),
            ]);

        // 2. update transaction(s)
      DB::table('transactions')
    ->where('note', 'like', '%W-' . $withdrawal->id . '%')
    ->update([
        'admin_status' => 'rejected', // ✅ allowed
        'status'       => 'failed',   // ✅ allowed
        'admin_note'   => $request->admin_note,
        'updated_at'   => now(),
    ]);


        // 3. refund amount back to user wallet
        DB::table('user_wallet')
            ->where('user_id', $withdrawal->user_id)
            ->update([
                'balance' => DB::raw('GREATEST(balance + ' . (float)$withdrawal->amount . ', 0)'),
                'updated_at' => now(),
            ]);
    });

    return back()->with('success', 'Withdrawal rejected and amount refunded to wallet.');
}
public function transactionsIndex(Request $request)
{
    // Admin session check
    if (!session()->has('admin_id')) {
        return redirect()->route('admin.login');
    }

    // Filter from URL: ?admin_status=approved | rejected
    $admin_status = $request->query('admin_status');

    // Fetch transactions from User model
    $transactions = User::fetchTransactions($admin_status);

    // Load common layout parts
    $pageHead = View::make('template.page_head')->render();
    $sidebar  = View::make('template.sidebar')->render();
    $pageFoot = View::make('template.page_foot')->render();

    // Load transactions view
    $table = View::make('users.transactions_list', compact('transactions', 'admin_status'))->render();

    return response($pageHead . $sidebar . $table . $pageFoot);
}
public function bookingsIndex(Request $request)
{
    if (!session()->has('admin_id')) {
        return redirect()->route('admin.login');
    }

    $filterKey   = $request->query('filterKey');
    $filterValue = $request->query('filterValue');

    $bookings = User::fetchBookings($filterKey, $filterValue);

    $pageHead = View::make('template.page_head')->render();
    $sidebar  = View::make('template.sidebar')->render();
    $pageFoot = View::make('template.page_foot')->render();

    // pass filter data to bookings_list so it can show active filter / clear link
    $table = View::make('users.bookings_list', compact('bookings', 'filterKey', 'filterValue'))->render();

    return response($pageHead . $sidebar . $table . $pageFoot);
}

    public function approveBookingPayment(Request $request, $id)
    {
        if (!session()->has('admin_id')) {
            return redirect()->route('admin.login');
        }

        $request->validate([
            'admin_note' => 'required|string'
        ]);

        $booking = User::fetchBookingById($id);
        if (!$booking) {
            return back()->with('error', 'Booking not found');
        }

        if ($booking->payment_status !== 'pending') {
            return back()->with('error', 'Payment is not pending');
        }

        $adminId = session('admin_id');

        // Update booking and record admin info
        User::updateBookingStatusWithAdmin($id, 'complete', 'approved', $adminId, $request->admin_note);

        // Optionally update transactions
        if (method_exists(User::class, 'updateTransactionsForBooking')) {
            User::updateTransactionsForBooking($booking, 'approved', $request->admin_note);
        }

        // Re-fetch booking if you need the newly saved admin fields (optional)
        $updatedBooking = User::fetchBookingById($id);

        // Build notification message including property/location
        $propertyDisplay = $updatedBooking->property_name ?? "Property #{$updatedBooking->property_id}";
        $addressParts = [];
        if (!empty($updatedBooking->locality_name)) $addressParts[] = $updatedBooking->locality_name;
        if (!empty($updatedBooking->city_name)) $addressParts[] = $updatedBooking->city_name;
        if (!empty($updatedBooking->state_name)) $addressParts[] = $updatedBooking->state_name;
        if (!empty($updatedBooking->property_address)) $addressParts[] = $updatedBooking->property_address;
        $fullAddress = implode(', ', $addressParts);

        $msg = "Your booking #{$updatedBooking->id} for {$propertyDisplay} ({$fullAddress}) payment has been approved. Note: " . $request->admin_note;
        if (method_exists(User::class, 'createNotification')) {
            User::createNotification($updatedBooking->user_id, $updatedBooking->property_id, $msg);
        }

        return back()->with('success', 'Booking payment approved');
    }

    public function rejectBookingPayment(Request $request, $id)
    {
        if (!session()->has('admin_id')) {
            return redirect()->route('admin.login');
        }

        $request->validate([
            'admin_note' => 'required|string'
        ]);

        $booking = User::fetchBookingById($id);
        if (!$booking) {
            return back()->with('error', 'Booking not found');
        }

        if ($booking->payment_status !== 'pending') {
            return back()->with('error', 'Payment is not pending');
        }

        $adminId = session('admin_id');

        User::updateBookingStatusWithAdmin($id, 'rejected', 'rejected', $adminId, $request->admin_note);

        if (method_exists(User::class, 'updateTransactionsForBooking')) {
            User::updateTransactionsForBooking($booking, 'rejected', $request->admin_note);
        }

        $updatedBooking = User::fetchBookingById($id);
        $propertyDisplay = $updatedBooking->property_name ?? "Property #{$updatedBooking->property_id}";
        $msg = "Your booking #{$updatedBooking->id} for {$propertyDisplay} payment has been rejected. Note: " . $request->admin_note;
        if (method_exists(User::class, 'createNotification')) {
            User::createNotification($updatedBooking->user_id, $updatedBooking->property_id, $msg);
        }

        return back()->with('success', 'Booking payment rejected');
    }

    public function editBooking($id)
{
    if (!session()->has('admin_id')) {
        return redirect()->route('admin.login');
    }

    $booking = User::fetchBookingById($id);

    if (!$booking) {
        return redirect()
            ->route('bookings.index')
            ->with('error', 'Booking not found');
    }

    // SAME STRUCTURE AS bookingsIndex
    $pageHead = View::make('template.page_head')->render();
    $sidebar  = View::make('template.sidebar')->render();
    $pageFoot = View::make('template.page_foot')->render();

    $content = View::make('users.booking_edit', compact('booking'))->render();

    return response($pageHead . $sidebar . $content . $pageFoot);
}

public function updateBooking(Request $request, $id)
{
    if (!session()->has('admin_id')) {
        return redirect()->route('admin.login');
    }

    $request->validate([
        'booking_status' => 'required|in:approved,completed,rejected',
        'check_in_time'  => 'required|date',
        'check_out_time' => 'required|date|after:check_in_time',
        'admin_note'     => 'required|string',
    ]);

    $booking = User::fetchBookingById($id);
    if (!$booking) {
        return redirect()->route('bookings.index')->with('error', 'Booking not found');
    }

    DB::table('my_bookings')->where('id', $id)->update([
        'booking_status'    => $request->booking_status,
        'check_in_time'     => $request->check_in_time,
        'check_out_time'    => $request->check_out_time,
        'admin_id'          => session('admin_id'),
        'admin_note'        => $request->admin_note,
        'admin_action_time' => now(),
        'updated_at'        => now(),
    ]);

    return redirect()
        ->route('bookings.index')
        ->with('success', 'Booking updated successfully');
}


public function couponsIndex()
{
    if (!session()->has('admin_id')) {
        return redirect()->route('admin.login');
    }

    $coupons = User::fetchAllCoupons();

    return response(
        View::make('template.page_head')->render() .
        View::make('template.sidebar')->render() .
        View::make('users.coupons_list', compact('coupons'))->render() .
        View::make('template.page_foot')->render()
    );
}
public function couponsCreate()
{
    return response(
        View::make('template.page_head')->render() .
        View::make('template.sidebar')->render() .
        View::make('users.coupons_form')->render() .
        View::make('template.page_foot')->render()
    );
}
public function couponsStore(Request $request)
{
    $request->validate([
        'name'        => 'required|string|max:255',
        'value'       => 'required|integer|min:0',
        'type'        => 'required|in:flat,percentage',
        'status'      => 'required|in:active,inactive,used',
        'expiry_date' => 'nullable|date'
    ]);

    User::createCoupon([
        'admin_id'    => session('admin_id'), // admin_subadmin.id
        'name'        => $request->name,
        'coupon_code' => $this->generateCouponCode(rand(10, 20)),
        'expiry_date' => $request->expiry_date,
        'value'       => $request->value,
        'type'        => $request->type,
        'status'      => $request->status
    ]);

    return redirect()->route('coupons.index')->with('success', 'Coupon created');
}

public function couponsEdit($id)
{
    $coupon = User::fetchCouponById($id);

    return response(
        View::make('template.page_head')->render() .
        View::make('template.sidebar')->render() .
        View::make('users.coupons_form', compact('coupon'))->render() .
        View::make('template.page_foot')->render()
    );
}
public function couponsUpdate(Request $request, $id)
{
    $request->validate([
        'name'        => 'required|string|max:255',
        'value'       => 'required|integer|min:0',
        'type'        => 'required|in:flat,percentage',
        'status'      => 'required|in:active,inactive,used',
        'expiry_date' => 'nullable|date'
    ]);

    User::updateCoupon($id, $request->all());

    return redirect()->route('coupons.index')->with('success', 'Coupon updated');
}

public function couponsDestroy($id)
{
    User::deleteCoupon($id);
    return redirect()->route('coupons.index')->with('success','Coupon deactivated');
}

private function generateCouponCode($length = 12)
{
    $length = max(10, min($length, 20)); // enforce 10–20
    return strtoupper(substr(str_shuffle(
        'ABCDEFGHIJKLMNOPQRSTUVWXYZ0123456789'
    ), 0, $length));
}



}
