<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Facades\DB;

class state extends Model
{
    protected $table = 'states';
    protected $primaryKey = 'id';
    public $timestamps = true;

    protected $fillable = ['name','status'];

    // ---------------- States ----------------
    public static function fetchAllStates()
    {
        return self::orderBy('id', 'desc')->get();
    }

    public static function fetchStateById($id)
    {
        return self::find($id);
    }

    public static function createState($data)
    {
        return self::create([
            'name'   => $data['name'] ?? null,
            'status' => $data['status'] ?? 'active',
        ]);
    }

    public static function updateState($id, $data)
    {
        $state = self::find($id);
        if (!$state) return null;
        $state->update([
            'name'   => $data['name'] ?? $state->name,
            'status' => $data['status'] ?? $state->status,
        ]);
        return $state;
    }

    // ---------------- Cities ----------------
    public static function fetchAllCities()
    {
        return DB::table('cities')
            ->join('states','cities.state_id','=','states.id')
            ->select('cities.*','states.name as state_name')
            ->orderBy('cities.id','desc')
            ->get();
    }

    public static function fetchCityById($id)
    {
        return DB::table('cities')->where('id',$id)->first();
    }

    public static function createCity($data)
    {
        return DB::table('cities')->insert([
            'state_id' => $data['state_id'],
            'name'     => $data['name'],
            'status'   => $data['status'] ?? 'active',
            'created_at' => now(),
            'updated_at' => now(),
        ]);
    }

    public static function updateCity($id,$data)
    {
        return DB::table('cities')->where('id',$id)->update([
            'state_id' => $data['state_id'],
            'name'     => $data['name'],
            'status'   => $data['status'],
            'updated_at' => now(),
        ]);
    }

    // ---------------- Localities ----------------
    public static function fetchAllLocalities()
    {
        return DB::table('localities')
            ->join('cities','localities.city_id','=','cities.id')
            ->join('states','cities.state_id','=','states.id')
            ->select('localities.*','cities.name as city_name','states.name as state_name')
            ->orderBy('localities.id','desc')
            ->get();
    }

    public static function fetchLocalityById($id)
    {
        return DB::table('localities')->where('id',$id)->first();
    }

    public static function createLocality($data)
    {
        return DB::table('localities')->insert([
            'city_id'  => $data['city_id'],
            'name'     => $data['name'],
            'status'   => $data['status'] ?? 'active',
            'created_at' => now(),
            'updated_at' => now(),
        ]);
    }

    public static function updateLocality($id,$data)
    {
        return DB::table('localities')->where('id',$id)->update([
            'city_id'  => $data['city_id'],
            'name'     => $data['name'],
            'status'   => $data['status'],
            'updated_at' => now(),
        ]);
    }

    // ---------------- Extra Services ----------------
    // Note: these use the same DB-style pattern as cities/localities.
    public static function fetchAllExtraServices()
    {
        $services = DB::table('extra_services')
            ->orderBy('id','desc')
            ->get();

        // add small presentation helpers so the view can use ->formatted_price and ->price_type_label
        return $services->map(function($s) {
            $s->formatted_price = '₹' . number_format($s->price, 2);
            $s->price_type_label = ($s->price_type === 'per_day') ? 'Per Day' : 'One Time';
            return $s;
        });
    }

    public static function fetchExtraServiceById($id)
    {
        $s = DB::table('extra_services')->where('id',$id)->first();
        if ($s) {
            $s->formatted_price = '₹' . number_format($s->price, 2);
            $s->price_type_label = ($s->price_type === 'per_day') ? 'Per Day' : 'One Time';
        }
        return $s;
    }

    public static function createExtraService($data)
    {
        return DB::table('extra_services')->insert([
            'name'       => $data['name'],
            'price'      => $data['price'] ?? 0,
            'price_type' => $data['price_type'] ?? 'one_time',
            'status'     => $data['status'] ?? 'active',
            'description'=> $data['description'] ?? null,
            'created_at' => now(),
            'updated_at' => now(),
        ]);
    }

    public static function updateExtraService($id, $data)
    {
        return DB::table('extra_services')->where('id',$id)->update([
            'name'       => $data['name'],
            'price'      => $data['price'],
            'price_type' => $data['price_type'],
            'status'     => $data['status'],
            'description'=> $data['description'] ?? null,
            'updated_at' => now(),
        ]);
    }

    public static function deactivateExtraService($id)
    {
        return DB::table('extra_services')->where('id',$id)->update([
            'status' => 'inactive',
            'updated_at' => now(),
        ]);
    }

    // optional: permanently delete (if you ever need)
    public static function deleteExtraService($id)
    {
        return DB::table('extra_services')->where('id',$id)->delete();
    }


    // ---------------- Location helpers (for properties form) ----------------

    /**
     * Fetch states ordered by name (for dropdowns)
     */
    public static function fetchStates()
    {
        return DB::table('states')->orderBy('name')->get();
    }

    /**
     * Fetch cities for a given state_id
     */
    public static function fetchCitiesByState($state_id)
    {
        return DB::table('cities')->where('state_id', $state_id)->orderBy('name')->get();
    }

    /**
     * Fetch localities for a given city_id
     */
    public static function fetchLocalitiesByCity($city_id)
    {
        return DB::table('localities')->where('city_id', $city_id)->orderBy('name')->get();
    }


    // ---------------- Properties ----------------

    /**
     * Fetch all properties with state/city/locality names
     */
    public static function fetchAllProperties()
    {
        return DB::table('properties as p')
            ->leftJoin('states as s', 'p.state_id', 's.id')
            ->leftJoin('cities as c', 'p.city_id', 'c.id')
            ->leftJoin('localities as l', 'p.locality_id', 'l.id')
            ->select('p.*', 's.name as state_name', 'c.name as city_name', 'l.name as locality_name')
            ->orderBy('p.created_at', 'desc')
            ->get();
    }

    /**
     * Fetch a single property with images array
     */
    public static function fetchPropertyById($id)
    {
        $property = DB::table('properties as p')
            ->leftJoin('states as s', 'p.state_id', 's.id')
            ->leftJoin('cities as c', 'p.city_id', 'c.id')
            ->leftJoin('localities as l', 'p.locality_id', 'l.id')
            ->select('p.*', 's.name as state_name', 'c.name as city_name', 'l.name as locality_name')
            ->where('p.id', $id)
            ->first();

        if ($property) {
            $property->images = DB::table('property_images')->where('property_id', $id)->get();
        }

        return $property;
    }

    /**
     * Create property and return inserted id
     */
    public static function createProperty($data)
    {
        return DB::table('properties')->insertGetId(array_merge($data, [
            'created_at' => now(),
            'updated_at' => now()
        ]));
    }

    /**
     * Update property
     */
    public static function updateProperty($id, $data)
    {
        return DB::table('properties')->where('id', $id)->update(array_merge($data, [
            'updated_at' => now()
        ]));
    }

    /**
     * Deactivate (soft) property
     */
    public static function deactivateProperty($id)
    {
        return DB::table('properties')->where('id', $id)->update(['status' => 'inactive', 'updated_at' => now()]);
    }


    // ---------------- Property images ----------------

    public static function savePropertyImage($propertyId, $filename)
    {
        return DB::table('property_images')->insert([
            'property_id' => $propertyId,
            'filename' => $filename,
            'created_at' => now(),
            'updated_at' => now()
        ]);
    }

    public static function countPropertyImages($propertyId)
    {
        return DB::table('property_images')->where('property_id', $propertyId)->count();
    }


    // ---------------- Property <-> Extra Services pivot ----------------

    public static function attachServicesToProperty($propertyId, $serviceIds = [])
    {
        foreach ($serviceIds as $sid) {
            DB::table('property_extra_service')->insertOrIgnore([
                'property_id' => $propertyId,
                'service_id' => $sid,
                'created_at' => now(),
                'updated_at' => now()
            ]);
        }
    }

    public static function syncServicesToProperty($propertyId, $serviceIds = [])
    {
        DB::table('property_extra_service')->where('property_id', $propertyId)->delete();
        if (!empty($serviceIds)) {
            foreach ($serviceIds as $sid) {
                DB::table('property_extra_service')->insert([
                    'property_id' => $propertyId,
                    'service_id' => $sid,
                    'created_at' => now(),
                    'updated_at' => now()
                ]);
            }
        }
    }

    public static function fetchServicesForProperty($propertyId)
    {
        return DB::table('property_extra_service as pes')
            ->join('extra_services as es','pes.service_id','es.id')
            ->where('pes.property_id', $propertyId)
            ->pluck('service_id')
            ->toArray();
    }

}
