<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Facades\DB;

class User extends Model
{
    protected $table = 'users';

    // Fetch all users, optionally filter by user_type
    public static function fetchAllUsers($user_type = null)
    {
        $query = DB::table('users')->orderBy('user_id', 'desc');

        if (!empty($user_type) && in_array($user_type, ['user', 'merchant'])) {
            $query->where('user_type', $user_type);
        }

        return $query->get();
    }

    // Fetch single user by id
    public static function fetchUserById($id)
    {
        return DB::table('users')->where('id', $id)->first();
    }

    // Insert new user
    public static function createUser($data)
    {
        return DB::table('users')->insert([
            'name' => $data['name'],
            'email' => $data['email'],
            'phone_no' => $data['phone_no'] ?? null,
            'status' => $data['status'] ?? 'active',
            'user_type' => $data['user_type'] ?? 'user', // default to user
            'created_at' => now(),
            'updated_at' => now()
        ]);
    }

    // Update user
    public static function updateUser($id, $data)
    {
        return DB::table('users')->where('id', $id)->update([
            'name' => $data['name'],
            'email' => $data['email'],
            'phone_no' => $data['phone_no'] ?? null,
            'status' => $data['status'] ?? 'active',
            'user_type' => $data['user_type'] ?? 'user',
            'updated_at' => now()
        ]);
    }

    // Soft-delete user (set inactive)
    public static function deleteUserById($id)
    {
        return DB::table('users')->where('id', $id)->update(['status' => 'inactive']);
    }



    // Fetch withdrawals with optional status filter
public static function fetchWithdrawals($status = null)
{
    $query = DB::table('withdrawals')
        ->select(
            'withdrawals.*',
            'users.name as user_name',
            'users.email as user_email'
        )
        ->leftJoin('users', 'users.user_id', '=', 'withdrawals.user_id')
        ->orderBy('withdrawals.id', 'desc');

    if (!empty($status) && in_array($status, ['pending','approved','rejected'])) {
        $query->where('withdrawals.status', $status);
    }

    return $query->get();
}

// Fetch single withdrawal
public static function fetchWithdrawalById($id)
{
    return DB::table('withdrawals')->where('id', $id)->first();
}

// Update withdrawal status
public static function updateWithdrawalStatus($id, $status)
{
    return DB::table('withdrawals')
        ->where('id', $id)
        ->update([
            'status' => $status,
            'updated_at' => now()
        ]);
}
/* ===================== Withdrawals ===================== */

public static function updateWithdrawalWithNote($id, $status, $adminNote)
{
    return DB::table('withdrawals')
        ->where('id', $id)
        ->update([
            'status' => $status,
            'admin_note' => $adminNote,
            'updated_at' => now()
        ]);
}

/* ===================== Transactions ===================== */

public static function createTransaction($data)
{
    return DB::table('transactions')->insert([
        'user_id' => $data['user_id'],
        'hideout_id' => $data['hideout_id'] ?? null,
        'status' => $data['status'] ?? 'pending',
        'admin_status' => $data['admin_status'] ?? null,
        'admin_note' => $data['admin_note'] ?? null,
        'note' => $data['note'] ?? null,
        'created_at' => now(),
        'updated_at' => now()
    ]);
}

public static function fetchTransactions($admin_status = null)
{
    $query = DB::table('transactions')
        ->select(
            'transactions.*',
            'users.name as user_name',
            'users.email as user_email',
            'properties.name as hideout_name'   // <-- added
        )
        ->leftJoin('users', 'users.user_id', '=', 'transactions.user_id')
        ->leftJoin('properties', 'properties.id', '=', 'transactions.hideout_id') // <-- added
        ->orderBy('transactions.id', 'desc');

    if (!empty($admin_status) && in_array($admin_status, ['approved','rejected'])) {
        $query->where('transactions.admin_status', $admin_status);
    }

    return $query->get();
}

public static function updateTransactionAdminAction($userId, $status, $note)
{
    return DB::table('transactions')
        ->where('user_id', $userId)
        ->update([
            'admin_status' => $status,
            'admin_note' => $note,
            'updated_at' => now()
        ]);
}


/* ---------------- Bookings ----------------- */

/**
 * Fetch bookings with optional filters.
 * $filter can be booking_status (pending/approved/completed) or payment_status (pending/complete).
 */
public static function fetchBookings($filterKey = null, $filterValue = null)
{
    $query = DB::table('my_bookings')
        ->select(
            'my_bookings.id',
            'my_bookings.user_id',
            'my_bookings.property_id',
            'my_bookings.check_in_time',
            'my_bookings.check_out_time',
            'my_bookings.payment_status',
            'my_bookings.booking_status',
            'my_bookings.created_at',
            'my_bookings.updated_at',
            'my_bookings.admin_id',
            'my_bookings.admin_note',
            'my_bookings.admin_action_time',

            // USER
            'users.name as user_name',
            'users.email as user_email',

            // PROPERTY
            'properties.name as property_name',
            'properties.address as property_address',
            'properties.price as property_price',
            'properties.price_unit as property_price_unit',

            // LOCATION
            'states.name as state_name',
            'cities.name as city_name',
            'localities.name as locality_name'
        )

        // ✅ FIXED JOIN
        ->leftJoin('users', 'users.user_id', '=', 'my_bookings.user_id')

        ->leftJoin('properties', 'properties.id', '=', 'my_bookings.property_id')
        ->leftJoin('states', 'states.id', '=', 'properties.state_id')
        ->leftJoin('cities', 'cities.id', '=', 'properties.city_id')
        ->leftJoin('localities', 'localities.id', '=', 'properties.locality_id')
        ->orderBy('my_bookings.id', 'desc');

    /* ---------------- FILTERING ---------------- */

    if (!empty($filterKey) && !is_null($filterValue)) {

        if ($filterKey === 'date') {
            // single date filter (overlapping bookings)
            try {
                $date = date('Y-m-d', strtotime($filterValue));
                $query->whereDate('my_bookings.check_in_time', '<=', $date)
                      ->whereDate('my_bookings.check_out_time', '>=', $date);
            } catch (\Exception $e) {
                // ignore invalid date
            }

        } elseif ($filterKey === 'date_range') {
            $parts = explode('|', $filterValue);
            if (count($parts) === 2) {
                $from = date('Y-m-d', strtotime($parts[0]));
                $to   = date('Y-m-d', strtotime($parts[1]));

                $query->whereDate('my_bookings.check_out_time', '>=', $from)
                      ->whereDate('my_bookings.check_in_time', '<=', $to);
            }

        } else {
            $query->where("my_bookings.$filterKey", $filterValue);
        }
    }

    return $query->get();
}


    public static function fetchBookingById($id)
    {
        return DB::table('my_bookings')
            ->select(
                // booking + admin fields
                'my_bookings.*',
                'my_bookings.admin_id',
                'my_bookings.admin_note',
                'my_bookings.admin_action_time',

                // user
                'users.name as user_name',
                'users.email as user_email',

                // property
                'properties.name as property_name',
                'properties.address as property_address',
                'properties.price as property_price',
                'properties.price_unit as property_price_unit',

                // location names
                'states.id as state_id',
                'states.name as state_name',
                'cities.id as city_id',
                'cities.name as city_name',
                'localities.id as locality_id',
                'localities.name as locality_name'
            )
            ->leftJoin('users', 'users.user_id', '=', 'my_bookings.user_id')
            ->leftJoin('properties', 'properties.id', '=', 'my_bookings.property_id')
            ->leftJoin('states', 'states.id', '=', 'properties.state_id')
            ->leftJoin('cities', 'cities.id', '=', 'properties.city_id')
            ->leftJoin('localities', 'localities.id', '=', 'properties.locality_id')
            ->where('my_bookings.id', $id)
            ->first();
    }

    public static function updateBookingStatusWithAdmin($bookingId, $paymentStatus = null, $bookingStatus = null, $adminId = null, $adminNote = null)
    {
        $data = [];

        if (!is_null($paymentStatus)) {
            $data['payment_status'] = $paymentStatus;
        }
        if (!is_null($bookingStatus)) {
            $data['booking_status'] = $bookingStatus;
        }

        if (!is_null($adminId)) {
            $data['admin_id'] = $adminId;
            $data['admin_note'] = $adminNote;
            $data['admin_action_time'] = Carbon::now();
        }

        if (empty($data)) {
            return false;
        }

        return DB::table('my_bookings')->where('id', $bookingId)->update($data);
    }
public static function updateBookingPaymentStatus($id, $paymentStatus, $bookingStatus = null)
{
    $data = [
        'payment_status' => $paymentStatus,
        'updated_at' => now(),
    ];
    if (!is_null($bookingStatus)) {
        $data['booking_status'] = $bookingStatus;
    }
    return DB::table('my_bookings')->where('id', $id)->update($data);
}

/* ---------------- Transactions helper ----------------- */

/**
 * Update transaction records that relate to a booking (match by user_id & hideout/property id).
 * This will set admin_status and admin_note for matching transactions.
 */
public static function updateTransactionsForBooking($booking, $admin_status, $admin_note)
{
    if (!$booking) return false;

    return DB::table('transactions')
        ->where('user_id', $booking->user_id)
        ->where('hideout_id', $booking->property_id)
        ->update([
            'admin_status' => $admin_status,
            'admin_note' => $admin_note,
            'updated_at' => now()
        ]);
}

/* ---------------- Notifications ----------------- */

public static function createNotification($user_id, $hideout_id = null, $msg, $status = 'active')
{
    return DB::table('notifications')->insert([
        'user_id' => $user_id,
        'hideout_id' => $hideout_id,
        'msg' => $msg,
        'read' => 0,
        'status' => $status,
        'created_at' => now(),
        'updated_at' => now()
    ]);
}
public static function dashboardStats()
{
    $stats = [];

    // Users
    $stats['total_users'] = DB::table('users')->count();
    $stats['users_count'] = DB::table('users')->where('user_type', 'user')->count();
    $stats['merchants_count'] = DB::table('users')->where('user_type', 'merchant')->count();

    // Properties
    $stats['properties_total'] = DB::table('properties')->count();
    $stats['properties_active_count'] = DB::table('properties')->where('status', 'active')->count();
    $stats['properties_inactive_count'] = DB::table('properties')->where('status', 'inactive')->count();

    // Bookings
    $stats['bookings_total'] = DB::table('my_bookings')->count();
    $stats['bookings_active'] = DB::table('my_bookings')->where('status', 'active')->count();
    $stats['bookings_pending'] = DB::table('my_bookings')->where('booking_status', 'pending')->count();
    $stats['bookings_completed'] = DB::table('my_bookings')->where('booking_status', 'completed')->count();

    // Withdrawals
    $stats['withdrawals_total'] = DB::table('withdrawals')->count();
    $stats['withdrawals_pending'] = DB::table('withdrawals')->where('status', 'pending')->count();
    $stats['withdrawals_approved'] = DB::table('withdrawals')->where('status', 'approved')->count();

    // Tickets
    $stats['tickets_total'] = DB::table('tickets')->count();
    $stats['tickets_active'] = DB::table('tickets')->where('status', 'active')->count();
    $stats['tickets_closed'] = DB::table('tickets')->where('status', 'closed')->count();

    // ✅ Coupons
    $stats['coupons_total'] = DB::table('discount_coupons')->count();
    $stats['coupons_active'] = DB::table('discount_coupons')->where('status', 'active')->count();
    $stats['coupons_used'] = DB::table('discount_coupons')->where('status', 'used')->count();

    return $stats;
}

public static function fetchAllCoupons()
{
    return DB::table('discount_coupons')
        ->select(
            'discount_coupons.*',
            'admin_subadmin.name as admin_name',
            'admin_subadmin.gmail as admin_email',
            'users.name as used_by_name',
            'users.email as used_by_email'
        )
        ->leftJoin(
            'admin_subadmin',
            'admin_subadmin.id',
            '=',
            'discount_coupons.admin_id'
        )
        ->leftJoin(
            'users',
            'users.user_id',
            '=',
            'discount_coupons.used_by_user_id'
        )
        ->where('discount_coupons.status', '!=', 'inactive') // SOFT DELETE
        ->orderBy('discount_coupons.id', 'desc')
        ->get();
}

public static function fetchCouponById($id)
{
    return DB::table('discount_coupons')->where('id', $id)->first();
}

public static function createCoupon($data)
{
    return DB::table('discount_coupons')->insert([
        'admin_id'    => $data['admin_id'],
        'name'        => $data['name'],
        'coupon_code' => $data['coupon_code'],
        'expiry_date' => $data['expiry_date'] ?? null,
        'value'       => $data['value'],
        'type'        => $data['type'],
        'status'      => $data['status'],
        'created_at'  => now(),
        'updated_at'  => now()
    ]);
}

public static function updateCoupon($id, $data)
{
    return DB::table('discount_coupons')->where('id', $id)->update([
        'name'        => $data['name'],
        'expiry_date' => $data['expiry_date'] ?? null,
        'value'       => $data['value'],
        'type'        => $data['type'],
        'status'      => $data['status'],
        'updated_at'  => now()
    ]);
}

public static function deleteCoupon($id)
{
    return DB::table('discount_coupons')
        ->where('id', $id)
        ->update([
            'status' => 'inactive',
            'updated_at' => now()
        ]);
}


}
