{{-- resources/views/merchant_panel/property_form.blade.php --}}
@php
    use Illuminate\Support\Str;
@endphp

<div class="main-content">
  <div class="page-content">
    <div class="container-fluid">

      @if(session('success'))
        <div class="alert alert-success">{{ session('success') }}</div>
      @endif
      @if(session('error'))
        <div class="alert alert-danger">{{ session('error') }}</div>
      @endif

      @if($errors->any())
        <div class="alert alert-danger">
          <ul class="mb-0">
            @foreach($errors->all() as $err)
              <li>{{ $err }}</li>
            @endforeach
          </ul>
        </div>
      @endif

      <div class="card">
        <div class="card-header d-flex align-items-center">
          <h4 class="card-title mb-0 flex-grow-1">
            {{ $property ? 'Edit Property' : 'Submit Property' }}
          </h4>
        </div>

        <div class="card-body">
          <p class="text-muted">Fill the details below to {{ $property ? 'update' : 'list' }} your property / hideout.</p>

          {{-- Decide action + method depending on create/edit --}}
          @php
            $isEdit = !empty($property);
            $visibleSlug = $isEdit ? (Str::slug($property->name ?? 'property-' . $property->id) . '-' . $property->id) : null;
            $action = $isEdit ? route('merchant.properties.update', $visibleSlug) : route('merchant.properties.store');

            // existing images count for JS
            $existingImageCount = 0;
            if ($isEdit) {
                $existingImageCount = DB::table('property_images')->where('property_id', $property->id)->count();
            }
          @endphp

          <form id="propertyForm" class="row g-3 needs-validation" action="{{ $action }}" method="POST" enctype="multipart/form-data" novalidate>
            @csrf

            {{-- STATE --}}
            <div class="col-md-4">
              <label for="state_id" class="form-label">State</label>
              <select id="state_id" name="state_id" class="form-select" required>
                <option value="" selected disabled>Choose...</option>
                @foreach($states as $s)
                  <option value="{{ $s->id }}"
                    @if(old('state_id') && old('state_id') == $s->id) selected
                    @elseif($isEdit && $property->state_id == $s->id && !old('state_id')) selected
                    @endif
                  >{{ $s->name }}</option>
                @endforeach
              </select>
              <div class="invalid-feedback">Please select a state.</div>
            </div>

            {{-- CITY --}}
            <div class="col-md-4">
              <label for="city_id" class="form-label">City</label>
              <select id="city_id" name="city_id" class="form-select" required disabled>
                <option value="" selected disabled>Select state first</option>
              </select>
              <div class="invalid-feedback">Please select a city.</div>
            </div>

            {{-- LOCALITY --}}
            <div class="col-md-4">
              <label for="locality_id" class="form-label">Locality</label>
              <select id="locality_id" name="locality_id" class="form-select" required disabled>
                <option value="" selected disabled>Select city first</option>
              </select>
              <div class="invalid-feedback">Please select a locality.</div>
            </div>

            {{-- NAME --}}
            <div class="col-md-12">
              <label for="name" class="form-label">Name of place / hideout</label>
              <input type="text" id="name" name="name" class="form-control" maxlength="255" required
                     value="{{ old('name') ?? ($isEdit ? $property->name : '') }}">
              <div class="invalid-feedback">Please provide a name.</div>
            </div>

            {{-- ADDRESS --}}
            <div class="col-md-12">
              <label for="address" class="form-label">Address</label>
              <input type="text" id="address" name="address" class="form-control" maxlength="1000" required
                     value="{{ old('address') ?? ($isEdit ? $property->address : '') }}">
              <div class="invalid-feedback">Please provide an address.</div>
            </div>

            {{-- PRICE --}}
            <div class="col-md-4">
              <label for="price" class="form-label">Price</label>
              <input type="number" id="price" name="price" class="form-control" min="0" step="0.01" required
                     value="{{ old('price') ?? ($isEdit ? $property->price : '') }}">
              <div class="invalid-feedback">Please enter a valid price.</div>
            </div>

            {{-- PRICE UNIT --}}
            <div class="col-md-4">
              <label for="price_unit" class="form-label">Price unit</label>
              <select id="price_unit" name="price_unit" class="form-select" required>
                <option selected disabled value="">Choose...</option>
                <option value="per_hour" {{ (old('price_unit')=='per_hour' || ($isEdit && $property->price_unit=='per_hour' && !old('price_unit'))) ? 'selected' : '' }}>Per hour</option>
                <option value="per_day" {{ (old('price_unit')=='per_day' || ($isEdit && $property->price_unit=='per_day' && !old('price_unit'))) ? 'selected' : '' }}>Per day</option>
                <option value="one_time" {{ (old('price_unit')=='one_time' || ($isEdit && $property->price_unit=='one_time' && !old('price_unit'))) ? 'selected' : '' }}>One-time</option>
              </select>
              <div class="invalid-feedback">Please choose a price unit.</div>
            </div>

            {{-- FEATURES --}}
            <div class="col-md-12">
              <label class="form-label d-block">Facilities (features)</label>
              <input type="text" name="features" class="form-control" placeholder="e.g. Wifi,Free Parking,Snacks"
                     value="{{ old('features') ?? ($isEdit ? $property->features : '') }}">
              <div class="form-text">Enter comma-separated features. They will be stored as CSV.</div>
            </div>

            {{-- GOOGLE MAP IFRAME --}}
            <div class="col-md-12">
              <label for="map_iframe" class="form-label">
                Google Map Embed (iframe)
              </label>

              <textarea
                id="map_iframe"
                name="map_iframe"
                class="form-control"
                rows="4"
                placeholder="Paste Google Maps iframe embed code here"
              >{{ old('map_iframe') ?? ($isEdit ? $property->map_iframe : '') }}</textarea>

              <div class="form-text">
                Go to Google Maps → Share → Embed a map → Copy iframe code.
              </div>
            </div>

            {{-- EXTRA SERVICES --}}
            <div class="col-md-12">
              <label class="form-label d-block">Extra Services</label>
              <div class="row">
                @php
                  $selectedServices = old('extra_services') ?? ($isEdit && $property->extra_services ? explode(',', $property->extra_services) : []);
                  if (!is_array($selectedServices)) $selectedServices = array_filter(array_map('trim', (array)$selectedServices));
                @endphp

                @foreach($extraServices as $es)
                  <div class="col-md-3">
                    <div class="form-check">
                      <input class="form-check-input" type="checkbox" name="extra_services[]" value="{{ $es->id }}"
                             id="es_{{ $es->id }}"
                             {{ in_array($es->id, $selectedServices) ? 'checked' : '' }}>
                      <label class="form-check-label" for="es_{{ $es->id }}">
                        {{ $es->name }} @if($es->price) ({{ $es->price }}) @endif
                      </label>
                    </div>
                  </div>
                @endforeach
              </div>
            </div>

            {{-- IMAGE UPLOAD --}}
            <div class="col-md-12">
              <label for="images" class="form-label">Images (Minimum 5 required, Maximum 10)</label>
              <input type="file" id="images" name="images[]" class="form-control" accept="image/*" multiple {{ $isEdit ? '' : 'required' }}>
              <div class="form-text">JPEG/PNG/WebP. Max 5MB each. Existing images shown below for edit.</div>
              <div class="invalid-feedback" id="imagesInvalid">Please upload at least five images (max 10).</div>
            </div>

            {{-- EXISTING IMAGES (edit only) --}}
            @if($isEdit)
              <div class="col-md-12">
                <label class="form-label d-block">Existing Images</label>
                <div id="existingImages" class="d-flex flex-wrap gap-2">
                  @php
                    $images = DB::table('property_images')->where('property_id', $property->id)->orderBy('id')->get();
                    $imageBase = rtrim(env('PROPERTIES_IMAGE_PATH', ''), '/');
                  @endphp

                  @foreach($images as $img)
                    @php
                      if (!empty($img->filename) && preg_match('/^https?:\/\//i', $img->filename)) {
                          $imgUrl = $img->filename;
                      } elseif (!empty($img->filename) && !empty($imageBase)) {
                          $imgUrl = $imageBase . '/' . ltrim($img->filename, '/');
                      } elseif (!empty($img->filename)) {
                          $imgUrl = asset('storage/property_images/' . ltrim($img->filename, '/'));
                      } else {
                          $imgUrl = asset('assets/images/small/img-1.jpg');
                      }
                    @endphp

                    <div class="card p-1" style="width:150px;" data-img-id="{{ $img->id }}">
                      <img src="{{ $imgUrl }}" alt="" class="img-fluid mb-1" style="height:90px;object-fit:cover;width:100%;">
                      <div class="d-flex justify-content-between">
                        <button type="button" class="btn btn-sm btn-danger btn-delete-image" data-image-id="{{ $img->id }}" data-filename="{{ $img->filename }}">Delete</button>
                        <a href="{{ $imgUrl }}" target="_blank" class="btn btn-sm btn-secondary">View</a>
                      </div>
                    </div>
                  @endforeach
                </div>
              </div>
            @endif

            <div class="col-12">
              <button class="btn btn-primary" type="submit">{{ $isEdit ? 'Update Property' : 'Submit Property' }}</button>
              <a href="{{ route('merchant.index') }}" class="btn btn-secondary">Cancel</a>
            </div>
          </form>
        </div>
      </div>

    </div>
  </div>
</div>

{{-- SCRIPTS --}}
<script>
document.addEventListener('DOMContentLoaded', function() {
  const stateEl = document.getElementById('state_id');
  const cityEl  = document.getElementById('city_id');
  const localityEl = document.getElementById('locality_id');
  const imagesInput = document.getElementById('images');
  const imagesInvalid = document.getElementById('imagesInvalid');
  const form = document.getElementById('propertyForm');

  // expose existing count to JS
  window.existingImageCount = {{ (int)$existingImageCount }};

  async function fetchJson(url, opts = {}) {
    const res = await fetch(url, Object.assign({ headers: { 'X-Requested-With': 'XMLHttpRequest' } }, opts));
    if (!res.ok) throw new Error('Network error');
    return await res.json();
  }

  async function loadCities(stateId, selectedCityId = null) {
    cityEl.innerHTML = '<option selected disabled>Loading...</option>';
    cityEl.disabled = true;
    localityEl.innerHTML = '<option selected disabled value="">Select city first</option>';
    localityEl.disabled = true;
    try {
      let url = "{{ route('merchant.ajax.cities') }}";
      if (stateId) url += '?state_id=' + encodeURIComponent(stateId);
      const data = await fetchJson(url);
      cityEl.innerHTML = '<option selected disabled value="">Choose...</option>';
      if (Array.isArray(data) && data.length) {
        data.forEach(c => {
          const o = document.createElement('option');
          o.value = c.id;
          o.textContent = c.name;
          cityEl.appendChild(o);
        });
        cityEl.disabled = false;
        if (selectedCityId) cityEl.value = selectedCityId;
      } else {
        cityEl.innerHTML = '<option selected disabled value="">No cities</option>';
        cityEl.disabled = true;
      }
    } catch (e) {
      cityEl.innerHTML = '<option selected disabled value="">Error loading</option>';
      cityEl.disabled = true;
    }
  }

  async function loadLocalities(cityId, selectedLocalityId = null) {
    localityEl.innerHTML = '<option selected disabled>Loading...</option>';
    localityEl.disabled = true;
    try {
      let url = "{{ route('merchant.ajax.localities') }}";
      if (cityId) url += '?city_id=' + encodeURIComponent(cityId);
      const data = await fetchJson(url);
      localityEl.innerHTML = '<option selected disabled value="">Choose...</option>';
      if (Array.isArray(data) && data.length) {
        data.forEach(l => {
          const o = document.createElement('option');
          o.value = l.id;
          o.textContent = l.name;
          localityEl.appendChild(o);
        });
        localityEl.disabled = false;
        if (selectedLocalityId) localityEl.value = selectedLocalityId;
      } else {
        localityEl.innerHTML = '<option selected disabled value="">No localities</option>';
        localityEl.disabled = true;
      }
    } catch (e) {
      localityEl.innerHTML = '<option selected disabled value="">Error loading</option>';
      localityEl.disabled = true;
    }
  }

  // events
  stateEl?.addEventListener('change', function() { loadCities(this.value); });
  cityEl?.addEventListener('change', function() { loadLocalities(this.value); });

  // images validation (1–10, with minimum 5 total)
  function validateImages() {
    const files = imagesInput?.files || [];
    const isEdit = {{ $isEdit ? 'true' : 'false' }};
    const existing = window.existingImageCount || 0;
    const total = isEdit ? (existing + files.length) : files.length;

    if (total < 5) {
      imagesInput.classList.add('is-invalid');
      imagesInvalid.textContent = 'Minimum 5 images are required. Currently selected/available: ' + total;
      return false;
    }
    if (files.length > 10) {
      imagesInput.classList.add('is-invalid');
      imagesInvalid.textContent = 'Maximum 10 images are allowed.';
      return false;
    }
    imagesInput.classList.remove('is-invalid');
    return true;
  }
  imagesInput?.addEventListener('change', validateImages);

  form?.addEventListener('submit', function(e) {
    if (!form.checkValidity() || !validateImages()) {
      e.preventDefault();
      e.stopPropagation();
    }
    form.classList.add('was-validated');
  }, false);

  // populate selects if old() or property exists
  const oldState = "{{ old('state_id') }}";
  const oldCity  = "{{ old('city_id') }}";
  const oldLocal = "{{ old('locality_id') }}";
  const propState = "{{ $isEdit ? $property->state_id : '' }}";
  const propCity  = "{{ $isEdit ? $property->city_id : '' }}";
  const propLocal = "{{ $isEdit ? $property->locality_id : '' }}";

  (async function init() {
    if (oldState || propState) {
      const s = oldState || propState;
      await loadCities(s, oldCity || propCity || null);
      if (oldCity || propCity) {
        const c = oldCity || propCity;
        await loadLocalities(c, oldLocal || propLocal || null);
      }
    } else {
      // do nothing (keeps dependent selects disabled)
    }
  })();

  // DELETE EXISTING IMAGE (AJAX)
  document.querySelectorAll('.btn-delete-image').forEach(btn => {
    btn.addEventListener('click', async function () {
      const imageId = this.dataset.imageId;
      if (!confirm('Delete this image?')) return;

      try {
        const url = "{{ url('merchant/properties/image') }}/" + encodeURIComponent(imageId) + "/delete";
        const res = await fetch(url, { method: 'POST', headers: {
          'X-Requested-With': 'XMLHttpRequest',
          'X-CSRF-TOKEN': '{{ csrf_token() }}',
          'Content-Type': 'application/json'
        }});
        if (!res.ok) throw new Error('Network error');
        const json = await res.json();
        if (json.success) {
          const card = document.querySelector('[data-img-id="'+imageId+'"]');
          if (card) card.remove();
          // decrement existing count in JS so validation remains correct
          window.existingImageCount = Math.max(0, (window.existingImageCount || 0) - 1);
        } else {
          alert(json.message || 'Could not delete image.');
        }
      } catch (e) {
        console.error(e);
        alert('Delete failed. See console for details.');
      }
    });
  });

});
</script>
